include_file("resource://com.presonus.musicdevices/sdk/midiprotocol.js");
include_file("resource://com.presonus.musicdevices/sdk/controlsurfacedevice.js");
include_file("AspProtocol.js");

class ColorLEDHandler extends PreSonus.ControlHandler 
{
    constructor(name, id) 
    {
        super();
        this.name = name;
        this.id = id;
    }

    sendValue(value, flags) 
    {
        let alpha = (value >> 24) & 0xFF;
        if (alpha == 0x00)
            value = 0x7FFFFFFF;
        let r = (value >> 1) & 0x7F;
        let g = (value >> 9) & 0x7F;
        let b = (value >> 17) & 0x7F;
        this.sendMidi(PreSonus.Midi.kNoteOn | 1, this.id, r);
        this.sendMidi(PreSonus.Midi.kNoteOn | 2, this.id, g);
        this.sendMidi(PreSonus.Midi.kNoteOn | 3, this.id, b);
    }
}

class LEDRingHandler extends PreSonus.ControlHandler 
{
    constructor(name, displayId) 
    {
        super();
        this.name = name;
        this.displayId = displayId;
    }

    sendValue(value, flags) 
    {
        let mode = Asp.ValueBar.kPan;

        if (flags & PreSonus.ControlValue.kDisabled)
            mode = Asp.ValueBar.kOff;
        else if (flags & PreSonus.ControlValue.kFill)
            mode = Asp.ValueBar.kFill;

        if (mode == Asp.ValueBar.kOff)
            this.sendMidi(PreSonus.Midi.kController + 1, 0x30 + this.displayId, 0);
        else if (mode == Asp.ValueBar.kFill)
            this.sendMidi(PreSonus.Midi.kController + 1, 0x30 + this.displayId, value * 127);
        else
            this.sendMidi(PreSonus.Midi.kController + 2, 0x30 + this.displayId, value * 127);

        // super.log("id: " + this.displayId)
    }
}

class MeterHandler extends PreSonus.ControlHandler 
{
    constructor(name, displayId, meterId) 
    {
        super();
        this.name = name;
        this.displayId = displayId;
        this.meterId = meterId;
        this.lastValue = -1;
        this.peakMode = true;
    }

    sendMidiValue(value) 
    {
        this.sendMidi(this.meterId, value, 0);
    }

    sendValue(value, flags) 
    {
        let meterValue = Asp.Support.scaleMeter(value);
        if (this.peakMode) 
        {
            if (meterValue > 0 && meterValue > this.lastValue)
                this.sendMidiValue(meterValue);
        }
        else 
        {
            if (meterValue != this.lastValue)
                this.sendMidiValue(meterValue);
        }
        this.lastValue = meterValue;
    }

    idle() 
    {
        if (this.peakMode) 
        {
            if (this.lastValue == Asp.Support.kMaxMeter)
                this.sendMidiValue(this.lastValue);
        }
    }

    reset() 
    {
        this.sendMidiValue(0);
        this.lastValue = 0;
    }
}

class LevelMeterHandler extends PreSonus.ControlHandler 
{
    constructor(name, displayId, meterId, isRight) 
    {
        super();
        this.name = name;
        this.isRight = isRight;
        this.displayId = displayId;
        this.meterId = meterId;
        this.lastValue = -1;
    }

    idle() { }

    reset() {
        this.lastValue = 0;
    }

    sendMidiValue(value)
    {
        if (this.isRight)
            this.sendMidi(0xD0, (this.meterId << 4) | value, 0);
        else
            this.sendMidi(0xD1, (this.meterId << 4) | value, 0);
    }

    sendValue(value, flags) {
        let meterValue = Asp.Support.scaleMeter(value);
        if (this.peakMode) 
        {
            if (meterValue > 0 && meterValue > this.lastValue)
                this.sendMidiValue(meterValue);
        }
        else 
        {
            if (meterValue != this.lastValue)
                this.sendMidiValue(meterValue);
        }
        this.lastValue = meterValue;
    }
}

class TimeCodeHandler extends PreSonus.ControlHandler 
{
    constructor(name, displayId, lineId) {
        super();
        this.name = name;
        this.displayId = displayId;
        this.lineId = lineId;
        this.segments = ["", "", "", ""];
        this.labels = ["", "", "", ""];
        this.enabled = false;
        this.secondsLabels = ["hour", "min", "sec", "ms"];
        this.samplesLabels = ["", "", "", "samples"];
        this.musicalLabels = ["bar", "beat", "16th", "rem"];
        this.framesLabels = ["hour", "min", "sec", "frames"];
    }
    printSegment(value, size) {
        let valueString = value.toString();
        let result = "";
        let toPad = size - valueString.length;
        while (toPad > 0) {
            result += "0";
            toPad--;
        }
        result += valueString;
        return result;
    }
    updateSegment(index, text) {
        if (this.segments[index] != text) {
            this.segments[index] = text;
            if (this.enabled)
                this.sendSegment(index);
        }
    }
    sendSegment(index) {
        this.device.sendText(this.displayId + index, this.lineId, Asp.TextStyle.kCenter, this.segments[index]);
    }
    sendLabel(index) {
        this.device.sendText(this.displayId + index, this.lineId + 1, Asp.TextStyle.kCenter | Asp.TextStyle.kInvert, this.labels[index]);
    }
    setEnabled(state) {
        this.enabled = state;
        for (let i = 0; i < 4; i++) {
            this.device.enableTextCell(this.displayId + i, this.lineId, !state);
            this.device.enableTextCell(this.displayId + i, this.lineId + 1, !state);
            if (state) {
                this.sendSegment(i);
                this.sendLabel(i);
            }
        }
    }
    setFormat(format) {
        for (let i = 0; i < 4; i++) {
            switch (format) {
                case PreSonus.TimeFormat.kSamples:
                    this.labels[i] = this.samplesLabels[i];
                    break;
                case PreSonus.TimeFormat.kMusical:
                    this.labels[i] = this.musicalLabels[i];
                    break;
                case PreSonus.TimeFormat.kFrames:
                    this.labels[i] = this.framesLabels[i];
                    break;
                default:
                    this.labels[i] = this.secondsLabels[i];
                    break;
            }
            if (this.enabled)
                this.sendLabel(i);
        }
    }
    isUsingDisplay(displayId) {
        return displayId >= this.displayId && displayId < this.displayId + 4;
    }
    sendValue(value, flags) {
        if (value instanceof Int32Array) {
            let count = value[0];
            if (count == 4) {
                for (let i = 0; i < 4; i++) {
                    let text = this.printSegment(value[1 + i], value[1 + count + i]);
                    this.updateSegment(i, text);
                }
            }
            else if (count == 1) {
                let samplesString = this.printSegment(value[1], 10);
                let offset = 0;
                for (let i = 0; i < 4; i++) {
                    let size = i >= 2 ? 3 : 2;
                    let text = samplesString.slice(offset, offset + size);
                    this.updateSegment(i, text);
                    offset += size;
                }
            }
        }
    }
}

class TimeCodeOverlayHandler extends PreSonus.ControlHandler {
    constructor(name, timeCodeHandler) {
        super();
        this.name = name;
        this.timeCodeHandler = timeCodeHandler;
    }
    sendValue(value, flags) {
        this.timeCodeHandler.setEnabled(value);
    }
}

class TimeCodeFormatHandler extends PreSonus.ControlHandler {
    constructor(name, handler) {
        super();
        this.name = name;
        this.timeCodeHandler = handler;
    }
    sendValue(value, flags) {
        this.timeCodeHandler.setFormat(value);
    }
}

class DisplayModeHandler extends PreSonus.ControlHandler 
{
    constructor(name, displayId) 
    {
        super();
        this.name = name;
        this.displayId = displayId;
        this.mode = Asp.DisplayMode.kDefault;
    }

    sendValue(value, flags) 
    {
        this.mode = value;
        this.device.sendDisplayMode(this.displayId, value);
    }
}

class TextCellHandler extends PreSonus.ControlHandler 
{
    constructor(name, displayId, lineId, modeHandler) 
    {
        super();
        this.name = name;
        this.displayId = displayId;
        this.lineId = lineId;
        this.modeHandler = modeHandler;
        this.hostText = "";
        this.trimmedText = "";
        this.enabled = true;
    }

    sendValue(value, flags) 
    {
        this.hostText = value;
        this.updateTrimmedText();
        this.sendText();
    }

    updateTrimmedText() 
    {
        let maxLength = 12;

        if (this.lineId == 3)
            maxLength = 8;

        this.trimmedText = this.trimText(this.hostText, maxLength, true);
    }

    sendText() 
    {
        this.device.sendText(this.displayId, this.lineId, 0, this.trimmedText);
    }
}

class DisplayLineState 
{
    constructor(style, text, dirty) 
    {
        this.style = style;
        this.text = "";
        this.dirty = false;
    }
}

class DisplayBuffer 
{
    constructor(device, displayId) 
    {
        this.device = device;
        this.displayId = displayId;
        this.mode = Asp.DisplayMode.kDefault;
        this.modeChanged = false;
        this.lines = [];
        this.linesChanged = false;
        for (let line = 0; line < Asp.Support.kMaxLineCount; line++) {
            this.lines.push(new DisplayLineState(Asp.TextStyle.kCenter, "", false));
        }
    }

    setMode(mode) 
    {
        if (mode != this.mode) 
        {
            this.mode = mode;
            this.modeChanged = true;
            // this.device.log("setMode: " + this.mode)
        }      
    }

    setText(lineId, style, text) 
    {
        let state = this.lines[lineId];
        if (state.style != style || state.text != text) 
        {
            state.style = style;
            state.text = text;
            state.dirty = true;
            this.linesChanged = true;
        }
    }

    setDirty() 
    {
        this.modeChanged = true;
        this.linesChanged = true;
        for (let lineId in this.lines)
            this.lines[lineId].dirty = true;
    }

    update(device, sysexBuffer) 
    {
        let clear = false;

        if (this.linesChanged || this.modeChanged) 
        {
            for (let lineId = 0; lineId < this.lines.length; lineId++) 
            {
                let state = this.lines[lineId];

                if (state.dirty || clear) 
                {
                    // device.log("text update: " + this.displayId + "  line: " + lineId + "  " + state.text)
                    let sysexBufferDmy;

                    if (lineId == 3)
                    {
                        sysexBufferDmy = Asp.Support.buildDisplayTextEncoderFunctionSysex(sysexBuffer, this.displayId, this.lines.length, state.text)
                    }              
                    else if (lineId == 1) // Numbers
                    {
                        sysexBufferDmy = sysexBuffer;
                        sysexBufferDmy.begin(Asp.Support.kSysexHeaderAsp);
                        sysexBufferDmy.push(0x17);
                        sysexBufferDmy.push(0);

                        if (state.text == null || state.text == "")
                        {
                            device.channelNos[this.displayId] = 0;
                        }
                        else
                        {
                            var channelNoAsInt = parseInt(state.text);

                            if(isNaN(channelNoAsInt) || channelNoAsInt == 0 || channelNoAsInt > 127)
                                device.channelNos[this.displayId] = 0;
                            else
                                device.channelNos[this.displayId] = channelNoAsInt;
                        }

                        for (let i = 0; i < device.faderCount; i++)
                            sysexBufferDmy.push( device.channelNos[i]);

                        sysexBufferDmy.end();
                    }
                    else if (lineId == 2 || lineId == 0)
                    {
                        sysexBufferDmy = Asp.Support.buildDisplayTextSysex(device.sysexHeader, sysexBuffer, this.displayId, lineId, this.lines.length, state.text)
                    }
                    else
                        return;

                    device.sendSysex(sysexBufferDmy);                    
                    state.dirty = false;
                }
            }

            this.modeChanged = false;
            this.linesChanged = false;
        }
    }
}
class AspMidiDevice extends PreSonus.ControlSurfaceDevice 
{
    constructor(faderCount, sysexHeader) 
    {
        super();
        this.faderCount = faderCount;
        this.sysexHeader = sysexHeader;
        this.lastActiveSensing = 0;
    }

    onInit(hostDevice) {
        super.onInit(hostDevice);
        this.debugLog = true; // ###########################
        this.meters = [];
        this.textCells = [];
        this.ledRings = [];
        this.displayBuffers = [];
        this.channelNos = [0, 0, 0, 0, 0, 0, 0, 0];
        for (let i = 0; i < this.faderCount; i++) 
        {
            let displayBuffer = new DisplayBuffer(this, i);
            this.displayBuffers.push(displayBuffer);
            let displayModeHandler = new DisplayModeHandler("displayMode[" + i + "]", i);
            this.addHandler(displayModeHandler);
            this.textCells[i] = new Array;
            for (let line = 0; line < Asp.Support.kMaxLineCount; line++) 
            {
                let textHandler = new TextCellHandler("textCell[" + i + "][" + line + "]", i, line, displayModeHandler);
                this.textCells[i].push(textHandler);
                this.addHandler(textHandler);
            }
            let ledRingHandler = new LEDRingHandler("ledRing[" + i + "]", i);
            this.ledRings.push(ledRingHandler);
            this.addHandler(ledRingHandler);
            let rightMeterHandler = new LevelMeterHandler("meter[" + i + "][1]", i, i, false);
            let leftMeterHandler = new LevelMeterHandler("meter[" + i + "][0]", i, i, true);
            this.addHandler(leftMeterHandler);
            this.addHandler(rightMeterHandler);
            this.meters.push(leftMeterHandler);
        }
        this.timeCodeHandler = new TimeCodeHandler("timeCode", this.faderCount - 4, 0);
        this.addHandler(this.timeCodeHandler);
        this.addHandler(new TimeCodeOverlayHandler("timeCodeOverlay", this.timeCodeHandler));
        this.addHandler(new TimeCodeFormatHandler("timeCodeFormat", this.timeCodeHandler));
    }

    createHandler(name, attributes) 
    {
        let className = attributes.getAttribute("class");
        if (className == "ColorLEDHandler") 
        {
            let address = attributes.getAttribute("address");
            let handler = new ColorLEDHandler(name, address);
            this.addHandler(handler);
            return true;
        }
        return false;
    }

    enableTextCell(displayId, lineId, state) 
    {
        let textHandler = this.textCells[displayId][lineId];
        textHandler.enabled = state;
        if (state)
            textHandler.sendText();
    }

    sendDisplayMode(displayId, mode) 
    {
        this.displayBuffers[displayId].setMode(mode);
        let textHandler = this.textCells[displayId][3];

        switch (mode)
        {
            case 0:
                textHandler.sendValue("Volume");
                break;
            case 1:
                textHandler.sendValue("Plugin");
                break;
            case 2:
                textHandler.sendValue("Send");
                break;
            case 3:
                textHandler.sendValue("Pan");
                break;
            case 5:
                textHandler.sendValue("Cue");
                break;
            default:
                textHandler.sendValue(" ");
                break;
        }

        for (let i in this.meters)
            if (this.meters[i].displayId == displayId) 
            {
                let handler = this.meters[i];
            }
        if (this.timeCodeHandler.enabled && this.timeCodeHandler.isUsingDisplay(displayId))
        {
            let segmentId = displayId - this.timeCodeHandler.displayId;
            this.timeCodeHandler.sendSegment(segmentId);
            this.timeCodeHandler.sendLabel(segmentId);
        }
    }

    sendText(displayId, lineId, style, text) 
    {
        this.displayBuffers[displayId].setText(lineId, style, text);
    }

    onIdle(time) 
    {
        if (this.midiOutConnected)
        {
            for (let i in this.displayBuffers)
                this.displayBuffers[i].update(this, this.sysexSendBuffer);

            for (let i in this.meters)
                this.meters[i].idle();
        }
    }

    onMidiOutConnected(state) 
    {
        super.onMidiOutConnected(state);

        if (state)
        {
            this.resetDevice();

            // Set vol encoder to red
            this.sendMidi(PreSonus.Midi.kNoteOn | 1, 0x38, 0x7F);
            this.sendMidi(PreSonus.Midi.kNoteOn | 2, 0x38, 0x00);
            this.sendMidi(PreSonus.Midi.kNoteOn | 3, 0x38, 0x00);
        }
    }

    resetDevice() {
        for (let i in this.ledRings)
            this.ledRings[i].lastMode = Asp.ValueBar.kOff;
        this.hostDevice.invalidateAll();
        for (let i in this.displayBuffers)
            this.displayBuffers[i].setDirty();
    }
}

AspMidiDevice.kActiveSensingInterval = 1000;
function createAspDeviceInstance() {
    return new AspMidiDevice(8, Asp.Support.kSysexHeaderAsp);
}
