//
// Asparion 700
//
// www.asparion.de
// Copyright 2023
//
// v1.4

// TODO
//
// track nos -> not available yet
// stereo metering -> not available yet

//-----------------------------------------------------------------------------
// INCLUDE common functions
//-----------------------------------------------------------------------------

var midiremote_api = require('midiremote_api_v1')


//-----------------------------------------------------------------------------
// USER SETTINGS - settings can be adjusted by user
//-----------------------------------------------------------------------------

var extensionCount = 4;

var selectTrackWhenTouched = true
var displayMidiTracks = false
var visibleSendTracks = 5 // max 8
var showSendsOnFaders = true
var sendsFadersToZeroAtClose = true
var widerPanRing = false

// If no display is attached these can be set to false to improve performance
var vuEnabled = true
var hasDisplay = true


//-----------------------------------------------------------------------------
// VARS - global vars and consts
//-----------------------------------------------------------------------------

var VPOT_ASSIGN =
{
    TRACK : 40,
    SEND : 41,
    PAN : 42,
    FX : 43,
    EQ : 44,
    INSTRUMENT : 45
};

var PAGES =
{
    TRACK : 0,
    PAN : 1,
    FX : 2,
    EQ : 3,
    EQ2 : 4,
    QUICKCTRLS : 5,
    SEND0 : 6,  
};

var DISPLAY_WIDTH = 96;
var DISPLAY_WIDTH_THIRD = 64;
var SINGLE_DISPLAY_WIDTH = 12;
var SINGLE_DISPLAY_WIDTH_THIRD = 8;

var Extensions = [];
var vuMeterLastSend = [];
var activePage = -1;
var selectedTrackTitle = "";
var totalTrackCount = extensionCount * 8;

if (visibleSendTracks < 1)
    visibleSendTracks = 1
else if (visibleSendTracks > 8)
    visibleSendTracks = 8

    
//-----------------------------------------------------------------------------
// DRIVER SETUP - create driver object, midi ports and detection information
//-----------------------------------------------------------------------------

var deviceDriver

if (extensionCount > 1)
    deviceDriver = midiremote_api.makeDeviceDriver("Asparion", "D700FT & " +  (extensionCount - 1) + "x D700F", "www.asparion.de")
else
    deviceDriver = midiremote_api.makeDeviceDriver("Asparion", "D700FT", "www.asparion.de")

var midiInput = {}
var midiOutput = {}
var detectionUnit = deviceDriver.makeDetectionUnit()

function initPorts(extensionCount)
{
    for (var i = 0; i < extensionCount; i++)
    {
        if (i == 0)
        {
            midiInput[i] = deviceDriver.mPorts.makeMidiInput('D700FT Input')
            midiOutput[i] = deviceDriver.mPorts.makeMidiOutput('D700FT Output')

            detectionUnit.detectPortPair(midiInput[i],  midiOutput[i])
                .expectInputNameEquals('D 700')
                .expectOutputNameEquals('D 700')
            
            detectionUnit.detectPortPair(midiInput[i],  midiOutput[i])
                .expectInputNameStartsWith('D 700')
                .expectInputNameEndsWith(' 1')
                .expectOutputNameStartsWith('D 700')
                .expectOutputNameEndsWith(' 1')
        }
        else
        {       
            midiInput[i] = deviceDriver.mPorts.makeMidiInput('D700F ' + i + '  Input')
            midiOutput[i] = deviceDriver.mPorts.makeMidiOutput('D700F ' + i + '  Output')

            detectionUnit.detectPortPair(midiInput[i],  midiOutput[i])
                .expectInputNameEquals('MIDIIN' + (i + 1) + ' (D 700)')
                .expectOutputNameEquals('MIDIOUT' + (i + 1) + ' (D 700)')

            detectionUnit.detectPortPair(midiInput[i],  midiOutput[i])      
                .expectInputNameStartsWith('D 700')
                .expectInputNameEndsWith(' ' + (i + 1))
                .expectOutputNameStartsWith('D 700')
                .expectOutputNameEndsWith(' ' + (i + 1))
        }
    }
}


//-----------------------------------------------------------------------------
// CLASSES
//-----------------------------------------------------------------------------

function DisplayStrip(extensionID)
{
    this.extensionID = extensionID;
    this.channelNos = [];
    this.clearAllText()
}

DisplayStrip.prototype.setDisplayText = function(trackNo, row, text)
{
    this.text[trackNo][row] = text;
};

DisplayStrip.prototype.clearAllText = function()
{
    this.text = [];

    for (var i = 0; i < 8; i++) // tracks
    {
        this.text[i] = [];

        for (var ii = 0; ii < 3; ii++) // rows
        {
            this.text[i][ii] = " ";
        }
    }
};

DisplayStrip.prototype.clearTrackNos = function(activeDevice)
{
    this.channelNos = [];

    for (var i = 0; i < 8; i++) // tracks
    {
        this.channelNos[i] = 0;
    }

    this.wirteToLCDChannelNos(activeDevice);
};

DisplayStrip.prototype.clearAllTextInRow = function(row)
{
    for (var i = 0; i < 8; i++) // tracks
    {
        this.text[i][row] = " ";
    }
};

DisplayStrip.prototype.setTrackNo = function(activeDevice, trackNo, no)
{
    if (this.channelNos[trackNo] == no)
        return;

    this.channelNos[trackNo] = no;
    this.wirteToLCDChannelNos(activeDevice);
}

DisplayStrip.prototype.wirteToLCDChannelNos = function(activeDevice)
{
    var sysExMessage = [0xf0, 0x00, 0x00, 0x66, 0x14]; 

    sysExMessage.push(0x17);
    sysExMessage.push(0x00);

    for (var i = 0; i < 8; i++)
    {
        sysExMessage.push(this.channelNos[i] & 0xFF);
    }

    sysExMessage.push(0xf7)

    if (hasDisplay)
        midiOutput[this.extensionID].sendMidi(activeDevice, sysExMessage)
}

DisplayStrip.prototype.setText = function(activeDevice, dispNo, row, text)
{
    if (dispNo > 8 || row > 2)
        return

    var len = SINGLE_DISPLAY_WIDTH

    if (row == 2)
        len = SINGLE_DISPLAY_WIDTH_THIRD

    var position = dispNo * len;

    text = text.slice(0, len)

    if (this.text[dispNo][row] == text)
        return

    this.text[dispNo][row] = text

    this.writeToOLED(activeDevice, row, position, text, len)
}

DisplayStrip.prototype.writeToOLED = function(activeDevice, row, position, text, len)
{
    var sysExMessage = [0xf0, 0x00, 0x00, 0x66, 0x14]; 

    text = text.slice(0, len)
    
    if (row == 2)
    {
        sysExMessage.push(0x19);
        sysExMessage.push(position); 
    }
    else
    {
        sysExMessage.push(0x1A);
        sysExMessage.push(position); 
        sysExMessage.push(row + 1);
    }

    for(var i = 0; i < len; ++i)
        sysExMessage.push(text.charCodeAt(i))
    
    sysExMessage.push(0xf7)

    if (hasDisplay)
        midiOutput[this.extensionID].sendMidi(activeDevice, sysExMessage)
}

function EncoderStrip(extensionID)
{
	this.extensionID = extensionID;
	this.colors = [];

	for (var t = 0; t < 8; t++)
	{
		this.colors[t] = [];
        this.colors[t][0] = -1
        this.colors[t][1] = -1
        this.colors[t][2] = -1
	}

    this.values = [];
	this.ledMode = [];

	for ( var i = 0; i < 8; i++)
	{
		this.values[i] = -1;
		this.ledMode[i] = -1;
	}
}

EncoderStrip.prototype.sendValueToVpot = function(activeDevice, encoderNo, value, ledMode)
{
    sendMidiCC(activeDevice, this.extensionID, ledMode, 0x30 + encoderNo, value);
}

EncoderStrip.prototype.setLedRing = function(activeDevice,  encoderNo, value, ledMode)
{
    if (encoderNo > 8)
        return

    var newVal = Math.round(value * 127);

    if (newVal != this.values[encoderNo] || this.ledMode[encoderNo] != ledMode)
    {
        this.values[encoderNo] = newVal;
        this.ledMode[encoderNo] = ledMode;
    }
    else
        return;

    this.sendValueToVpot(activeDevice, encoderNo, this.values[encoderNo], this.ledMode[encoderNo]);
}

EncoderStrip.prototype.setColor = function(activeDevice, encoderNo, r, g, b)
{
    r = (r * 127) | 0;
    g = (g * 127) | 0;
    b = (b * 127) | 0;

	var changeFound = false;

	if (this.colors[encoderNo][0] != r)
	{
		this.colors[encoderNo][0] = r;
        sendMidiNoteOn(activeDevice, 1, this.extensionID, 0x20 + encoderNo, r);
		changeFound = true;
	}

	if (this.colors[encoderNo][1] != g)
	{
		this.colors[encoderNo][1] = g;
        sendMidiNoteOn(activeDevice, 2, this.extensionID, 0x20 + encoderNo, g)
		changeFound = true;
	}

	if (this.colors[encoderNo][2] != b || changeFound) // only refreshes after blue received
	{
		this.colors[encoderNo][2] = b;
        sendMidiNoteOn(activeDevice, 3, this.extensionID, 0x20 + encoderNo, b);
	}
}

function Extension(extensionID, activeDevice)
{
    this.extensionID = extensionID;
    this.displayStrip = new DisplayStrip(extensionID);
    this.encoderStrip = new EncoderStrip(extensionID);  
}

function ShortenText(text, maxChars)
{
    var dmyText = text.trim();

    if (dmyText.length > maxChars)
    {
        dmyText = dmyText.replace(/\s/g,'')
    }

    return dmyText
}


//-----------------------------------------------------------------------------
// FUNCS
//-----------------------------------------------------------------------------

function trackNoToExtensionID(trackNo)
{
    return (trackNo / 8) | 0;
}

function trackNoToTrackNo(trackNo)
{
    return trackNo % 8;
}

function setLedRing(activeDevice, trackNo, value, ledMode)
{
    Extensions[trackNoToExtensionID(trackNo)].encoderStrip.setLedRing(activeDevice, trackNoToTrackNo(trackNo), value, ledMode);
}

function writeText(activeDevice, trackNo, row, text)
{
    Extensions[trackNoToExtensionID(trackNo)].displayStrip.setText(activeDevice, trackNoToTrackNo(trackNo), row, text);
}

function setDisplayEncoderFunctionForPage(activeDevice, text)
{
    for (var exNo = 0; exNo < extensionCount; exNo++)
        for (var i = 0; i < 8; i++)
            Extensions[exNo].displayStrip.setText(activeDevice, i, 2, text);
}

function showDisplayTrackNo(activeDevice, trackNo, no)
{
    if (no < 0)
        return;    

    Extensions[trackNoToExtensionID(trackNo)].displayStrip.setTrackNo(activeDevice, trackNoToTrackNo(trackNo), no);       
}

function setColor(activeDevice, trackNo, r, g, b)
{
	Extensions[trackNoToExtensionID(trackNo)].encoderStrip.setColor(activeDevice, trackNoToTrackNo(trackNo), r, g, b);
}

function resetDev(activeDevice) 
{
    
    var data = [0xf0, 0x00, 0x00, 0x66, 0x14, 0x62, 0xf7]
    midiOutput[0].sendMidi(activeDevice, data)

    if (sendsFadersToZeroAtClose)
    {
        data = [0xf0, 0x00, 0x00, 0x66, 0x14, 0x61, 0xf7]
        midiOutput[0].sendMidi(activeDevice, data)
    }

    console.log('D700 reset')
}

function clearTrackNos(activeDevice) 
{
    for (var i = 0; i < extensionCount; i++)
    {
        Extensions[i].displayStrip.clearTrackNos(activeDevice)
    }
}

function clearDisplay(activeDevice) 
{
    for (var i = 0; i < extensionCount; i++)
    {
        Extensions[i].displayStrip.clearAllText()
    }

    var data = [0xf0, 0x00, 0x00, 0x66, 0x14, 0x42, 0xf7]
    midiOutput[0].sendMidi(activeDevice, data)
}

function clearDisplayRow(activeDevice, row) 
{
    if (row < 0 || row > 3)
        return;

    for (var i = 0; i < extensionCount; i++)
    {
        Extensions[i].displayStrip.clearAllTextInRow(row)
    }

    var data = [0xf0, 0x00, 0x00, 0x66, 0x14, 0x66 + row, 0xf7] // 66 to 69 for row 1 to 4
    midiOutput[0].sendMidi(activeDevice, data)
}

function sendMidiNoteOn(dev, channel, exID, ledID, val)
{   
    midiOutput[exID].sendMidi(dev, [ 0x90 + channel, ledID, val ])
}

function sendMidiChannelPressure(dev, exID, channel, val)
{
    midiOutput[exID].sendMidi(dev, [ 0xD0 + channel, val ])
}

function sendMidiPolyKeyPress(dev, exID, channel, ID, val)
{
    midiOutput[exID].sendMidi(dev, [ 0xA0 + channel, ID, val ])
}

function sendMidiPitchBend(dev, exID, channel, val)
{
    midiOutput[exID].sendMidi(dev, [ 0xe0 + channel, (val & 127), (val >> 7) & 127 ])
}

function sendMidiCC(dev, exID, channel, ID, val)
{
    midiOutput[exID].sendMidi(dev, [ 0xB0 + channel, ID, val ])
}

function makeButtonDisplayFeedback(button, exID, ledID) 
{
    button.mSurfaceValue.mOnProcessValueChange = function (activeDevice, newValue) 
    {
		sendMidiNoteOn(activeDevice, 0, this.exID, this.ledID, newValue)
	}.bind({exID, ledID})
}

function init()
{
    initPorts(extensionCount)
}

init()

deviceDriver.mOnDeactivate = function(activeDevice) 
{   
    resetDev(activeDevice)
    console.log('D700 deactivated')
}


deviceDriver.mOnActivate = function(activeDevice) 
{   
    console.log('D700 activated')
    resetDev(activeDevice) 

    for (var exNo = 0; exNo < extensionCount; exNo++)
    {
        Extensions[exNo] = new Extension(exNo, activeDevice);
    }

    // Set volume encoder to red
    sendMidiNoteOn(activeDevice, 1, 0, 56, 127);
    sendMidiNoteOn(activeDevice, 2, 0, 56, 0);
    sendMidiNoteOn(activeDevice, 3, 0, 56, 0);

    activatePage(activeDevice, PAGES.TRACK);
}


//-----------------------------------------------------------------------------
// SURFACE LAYOUT - create control elements and midi bindings
//-----------------------------------------------------------------------------

var surface = deviceDriver.mSurface

function bindMidiCC(button, exID, num) 
{
    button.mSurfaceValue.mMidiBinding.setInputPort(midiInput[exID]).bindToControlChange(0, num)
}

function bindNoteOn(button, exID, num) 
{
    button.mSurfaceValue.mMidiBinding.setInputPort(midiInput[exID]).bindToNote(0, num)
}

function makeFaderStrip(exID, channelIndex, x, y) 
{
    var faderStrip = {}
    var posFromTheTop = y

    faderStrip.knob = surface.makeKnob(x + 2 * channelIndex + 0.15, posFromTheTop, 1.7, 1.7)
    faderStrip.knob.mSurfaceValue.mMidiBinding.setInputPort(midiInput[exID]).bindToControlChange(0, 16 + channelIndex).setTypeRelativeSignedBit()
    
    faderStrip.knobClick = surface.makeButton(x + 2 * channelIndex - 0.1, posFromTheTop, 0.7, 0.7).setShapeCircle()
    bindNoteOn(faderStrip.knobClick, exID, 32 + channelIndex)

    posFromTheTop += 2
    faderStrip.btnMute = surface.makeButton(x + 2 * channelIndex, posFromTheTop + 0.2, 1, 0.8)
    faderStrip.btnSolo = surface.makeButton(x + 2 * channelIndex, posFromTheTop + 1, 1, 0.8)
    faderStrip.btnRec = surface.makeButton(x + 2 * channelIndex + 1, posFromTheTop + 0.2, 1, 0.8)
    faderStrip.btnSelect = surface.makeButton(x + 2 * channelIndex + 1, posFromTheTop + 1, 1, 0.8)

    faderStrip.btnMuteDbl = surface.makeButton(0, 0, 0.01, 0.01)
    faderStrip.btnSoloDbl = surface.makeButton(0, 0, 0.01, 0.01)
    faderStrip.btnRecDbl = surface.makeButton(0, 0, 0.01, 0.01)
    faderStrip.btnSelectDbl = surface.makeButton(0, 0, 0.01, 0.01)

    bindNoteOn(faderStrip.btnMute, exID, 16 + channelIndex)
    bindNoteOn(faderStrip.btnSolo, exID, 8 + channelIndex)
    bindNoteOn(faderStrip.btnRec, exID, 0 + channelIndex)
    bindNoteOn(faderStrip.btnSelect, exID, 24 + channelIndex)

    bindMidiCC(faderStrip.btnMuteDbl, exID, 32 + channelIndex)
    bindMidiCC(faderStrip.btnSoloDbl, exID, 8 + channelIndex)
    bindMidiCC(faderStrip.btnRecDbl, exID, 0 + channelIndex)
    bindMidiCC(faderStrip.btnSelectDbl, exID, 24 + channelIndex)
    
    posFromTheTop += 2.5
    faderStrip.fader = surface.makeFader(x + 2 * channelIndex + 0.3, posFromTheTop, 1.4, 11).setTypeVertical()

    if (selectTrackWhenTouched)
    {
        posFromTheTop += 11
        faderStrip.btnFaderTouch = surface.makeButton(x + 2 * channelIndex + 0.6, posFromTheTop, 0.8, 0.8).setShapeCircle()
        bindNoteOn(faderStrip.btnFaderTouch, exID, 104 + channelIndex)
    }

    faderStrip.fader.mSurfaceValue.mMidiBinding.setInputPort(midiInput[exID]).bindToPitchBend(channelIndex)
    // faderStrip.fader.mSurfaceValue.mMidiBinding.setTypeAbsolute()
    
    faderStrip.vPanSettable = surface.makeCustomValueVariable("vPanSettable")
    faderStrip.vVUMeter = surface.makeCustomValueVariable("vVUMeter")
    faderStrip.vVUMeterClip = surface.makeCustomValueVariable("vVUMeterClip")

    faderStrip.vSendsOns = {}

    faderStrip.displayLineLowerText = ""
    faderStrip.displayLineLowerTextPage = ""

    for(var i = 0; i < visibleSendTracks; i++)
    {
        faderStrip.vSendsOns[i] = surface.makeCustomValueVariable("vSendsOns" + i)
    }

    return faderStrip
}

function makeTransport(x, y) 
{
    var transport = {}
    var posFromTheTop = y

    posFromTheTop += 0.2

    transport.btnPan = surface.makeButton(x + 0.45, posFromTheTop, 1, 0.8)
    bindNoteOn(transport.btnPan, 0, 42)
    posFromTheTop += 0.8

    transport.btnEQ = surface.makeButton(x + 0.45, posFromTheTop, 1, 0.8)
    bindNoteOn(transport.btnEQ, 0, 44)
    posFromTheTop += 0.8

    transport.btnSend = surface.makeButton(x + 0.45, posFromTheTop, 1, 0.8)
    bindNoteOn(transport.btnSend, 0, 41)
    posFromTheTop += 0.8

    transport.btnFX = surface.makeButton(x + 0.45, posFromTheTop, 1, 0.8)
    bindNoteOn(transport.btnFX, 0, 43)
    posFromTheTop += 1

    transport.btnMagic = surface.makeButton(x + 0.3, posFromTheTop, 1.3, 1)
    bindNoteOn(transport.btnMagic, 0, 54)
    posFromTheTop += 1.3

    transport.btnMagicDbl = surface.makeButton(0, 0, 0.01, 0.01)
    bindNoteOn(transport.btnMagicDbl, 0, 55)

    transport.knob = surface.makeKnob(x + 0.1, posFromTheTop, 1.7, 1.7)
    transport.knob.mSurfaceValue.mMidiBinding.setInputPort(midiInput[0]).bindToControlChange(0, 0x70).setTypeRelativeSignedBit()
    transport.knobClick = surface.makeButton(x - 0.1, posFromTheTop, 0.7, 0.7).setShapeCircle()
    bindNoteOn(transport.knobClick, 0, 56)
    posFromTheTop += 2

    transport.knobDbl = surface.makeKnob(0, 0, 0.01, 0.01)
    transport.knobDbl.mSurfaceValue.mMidiBinding.setInputPort(midiInput[0]).bindToControlChange(0, 0x71).setTypeRelativeSignedBit()
    transport.knobClickDbl = surface.makeButton(0, 0, 0.01, 0.01).setShapeCircle()
    bindNoteOn(transport.knobClickDbl, 0, 57)


    transport.btnMetronome = surface.makeButton(x + 0.3, posFromTheTop, 1.3, 1)
    bindNoteOn(transport.btnMetronome, 0, 89)
    posFromTheTop += 1.1

    transport.btnCycle = surface.makeButton(x + 0.3, posFromTheTop, 1.3, 1)
    bindNoteOn(transport.btnCycle, 0, 86)
    posFromTheTop += 1.3

    transport.btnMetronomeDbl = surface.makeButton(0, 0, 0.01, 0.01)
    bindNoteOn(transport.btnMetronomeDbl, 0, 87)

    transport.btnCycleDbl = surface.makeButton(0, 0, 0.01, 0.01)
    bindNoteOn(transport.btnCycleDbl, 0, 82)


    transport.btnRecord = surface.makeButton(x, posFromTheTop, 1.9, 1.3)
    bindNoteOn(transport.btnRecord, 0, 95)
    posFromTheTop += 1.5

    transport.btnStart = surface.makeButton(x, posFromTheTop, 1.9, 1.3)
    bindNoteOn(transport.btnStart, 0, 94)
    posFromTheTop += 1.5

    transport.btnStop = surface.makeButton(x, posFromTheTop, 1.9, 1.3)
    bindNoteOn(transport.btnStop, 0, 93)
    posFromTheTop += 1.6


    transport.btnRecordDbl = surface.makeButton(0, 0, 0.01, 0.01)
    bindNoteOn(transport.btnRecordDbl, 0, 85)

    transport.btnStartDbl = surface.makeButton(0, 0, 0.01, 0.01)
    bindNoteOn(transport.btnStartDbl, 0, 84)

    transport.btnStopDbl = surface.makeButton(0, 0, 0.01, 0.01)
    bindNoteOn(transport.btnStopDbl, 0, 83)


    transport.btnBankLeft = surface.makeButton(x + 0.45, posFromTheTop, 1, 1.3)
    bindNoteOn(transport.btnBankLeft, 0, 46)
    posFromTheTop += 1.4

    transport.btnBankRight = surface.makeButton(x + 0.45, posFromTheTop, 1, 1.3)
    bindNoteOn(transport.btnBankRight, 0, 47)


    transport.btnBankLeftDbl = surface.makeButton(0, 0, 0.01, 0.01)
    bindNoteOn(transport.btnBankLeftDbl, 0, 48)

    transport.btnBankRightDbl = surface.makeButton(0, 0, 0.01, 0.01)
    bindNoteOn(transport.btnBankRightDbl, 0, 49)
    

    return transport
}

function makeVrituals()
{
    var virtuals = {}

    virtuals.SelectedTitle = surface.makeCustomValueVariable("SelectedTitle")

    virtuals.ActivatePageFaderVolume = surface.makeCustomValueVariable("ActivatePageFaderVolume")

    virtuals.ActivatePageVolume = surface.makeCustomValueVariable("ActivatePageVolume")
    virtuals.ActivatePagePan = surface.makeCustomValueVariable("ActivatePagePan")
    virtuals.ActivatePageEQ = surface.makeCustomValueVariable("ActivatePageEQ")
    virtuals.ActivatePageEQ2 = surface.makeCustomValueVariable("ActivatePageEQ2")
    virtuals.ActivatePageQuickCtrls = surface.makeCustomValueVariable("ActivatePageQuickCtrls")
    virtuals.ActivatePageFX = surface.makeCustomValueVariable("ActivatePageFX")

    virtuals.ActivatePageSends = {}
    virtuals.ActivatePageFaderSends = {}

    for(var i = 0; i < visibleSendTracks; i++)
    {
        virtuals.ActivatePageSends[i] = surface.makeCustomValueVariable("ActivatePageSend" + i)
        virtuals.ActivatePageFaderSends[i] = surface.makeCustomValueVariable("ActivatePageFaderSends" + i)
    }

    virtuals.EQOns = {}

    for(var i = 0; i < 4; i++)
    {
        virtuals.EQOns[i] = surface.makeCustomValueVariable("EQOns" + i)
    }

    return virtuals
}

function makeSurface() 
{
    var surfaceElements = {}

    surfaceElements.numStrips = extensionCount * 8
    surfaceElements.faderStrips = {}

    for(var exID = 0; exID < extensionCount; ++exID) 
    {
        for(var trackNo = 0; trackNo < 8; ++trackNo)
        {
            var globalTrackNo = exID * 8 + trackNo;
            surfaceElements.faderStrips[globalTrackNo] = makeFaderStrip(exID, trackNo, 0.4 + exID * 18, 1)
        }
    }

    surfaceElements.transport = makeTransport(extensionCount * 18, 0)
    surfaceElements.virtuals = makeVrituals()
    
    return surfaceElements
}

function ToggleValue(activeDevice, surfaceEle)
{
    var isOn = surfaceEle.getProcessValue(activeDevice) == 1
    surfaceEle.setProcessValue(activeDevice, isOn ? 0 : 1)
}

var surfaceElements = makeSurface()

makeButtonDisplayFeedback(surfaceElements.transport.btnBankLeft, 0, 46)
makeButtonDisplayFeedback(surfaceElements.transport.btnBankRight, 0, 47)
makeButtonDisplayFeedback(surfaceElements.transport.btnStop, 0, 93)
makeButtonDisplayFeedback(surfaceElements.transport.btnStart, 0, 94)
makeButtonDisplayFeedback(surfaceElements.transport.btnRecord, 0, 95)

makeButtonDisplayFeedback(surfaceElements.transport.btnCycle, 0, 86)
makeButtonDisplayFeedback(surfaceElements.transport.btnMetronome, 0, 89)
makeButtonDisplayFeedback(surfaceElements.transport.btnMagic, 0, 54)


//-----------------------------------------------------------------------------
// CREATE PAGES
//-----------------------------------------------------------------------------

function makeSubPage(subPageArea, pageNo, pageName) 
{
    var subPage = subPageArea.makeSubPage(pageName)
    var msgText = 'Sub page:' + pageNo + ' activated:' + pageName
    subPage.mOnActivate = function(activeDevice) 
    {
        // console.log(msgText)
        showPage(activeDevice, pageNo)
    }
    return subPage
}

function makeSubPageFader(subPageArea, pageNo, pageName) 
{
    var subPage = subPageArea.makeSubPage(pageName)
    return subPage
}

var page = deviceDriver.mMapping.makePage("Mixer")

var faderSubPageArea = page.makeSubPageArea("Faders")
var subPageFaderVolume = makeSubPageFader(faderSubPageArea, PAGES.TRACK, "Fader Volume")
var subPageFaderSends = {}

var knobSubPageArea = page.makeSubPageArea("Knobs")
var subPageVolume = makeSubPage(knobSubPageArea, PAGES.TRACK, "Volume")
var subPagePan = makeSubPage(knobSubPageArea, PAGES.PAN, "Pan")
var subPageEQ = makeSubPage(knobSubPageArea, PAGES.EQ, "EQ")
var subPageEQ2 = makeSubPage(knobSubPageArea, PAGES.EQ2, "EQ")
var subPageQuickCtrls = makeSubPage(knobSubPageArea, PAGES.QUICKCTRLS, "Quick Controls")
var subPageFX = makeSubPage(knobSubPageArea, PAGES.FX, "FX")
var subPageSends = {}

function makePageMixer() 
{
    page.makeActionBinding(surfaceElements.virtuals.ActivatePageFaderVolume, subPageFaderVolume.mAction.mActivate);

    page.makeActionBinding(surfaceElements.virtuals.ActivatePageVolume, subPageVolume.mAction.mActivate);
    page.makeActionBinding(surfaceElements.virtuals.ActivatePagePan, subPagePan.mAction.mActivate);
    page.makeActionBinding(surfaceElements.virtuals.ActivatePageEQ, subPageEQ.mAction.mActivate);
    page.makeActionBinding(surfaceElements.virtuals.ActivatePageEQ2, subPageEQ2.mAction.mActivate);
    page.makeActionBinding(surfaceElements.virtuals.ActivatePageQuickCtrls, subPageQuickCtrls.mAction.mActivate);
    page.makeActionBinding(surfaceElements.virtuals.ActivatePageFX, subPageFX.mAction.mActivate);

    for(var i = 0; i < visibleSendTracks; i++)
    {
        subPageSends[i] = makeSubPage(knobSubPageArea, PAGES.SEND0 + i, "Send " + (i + 1))
        page.makeActionBinding(surfaceElements.virtuals.ActivatePageSends[i], subPageSends[i].mAction.mActivate);

        subPageFaderSends[i] = makeSubPageFader(faderSubPageArea, PAGES.SEND0 + i, "Fader Send " + (i + 1))
        page.makeActionBinding(surfaceElements.virtuals.ActivatePageFaderSends[i], subPageFaderSends[i].mAction.mActivate);
    }

    page.makeValueBinding(surfaceElements.transport.btnStop.mSurfaceValue, page.mHostAccess.mTransport.mValue.mStop).setTypeToggle()
    page.makeValueBinding(surfaceElements.transport.btnStart.mSurfaceValue, page.mHostAccess.mTransport.mValue.mStart).setTypeToggle()    
    page.makeValueBinding(surfaceElements.transport.btnRecord.mSurfaceValue, page.mHostAccess.mTransport.mValue.mRecord).setTypeToggle()

    page.makeValueBinding(surfaceElements.transport.btnCycle.mSurfaceValue, page.mHostAccess.mTransport.mValue.mCycleActive).setTypeToggle()
    page.makeValueBinding(surfaceElements.transport.btnMetronome.mSurfaceValue, page.mHostAccess.mTransport.mValue.mMetronomeActive).setTypeToggle()

    // Magic can be set in Cubase
    //page.makeValueBinding(surfaceElements.transport.btnMagic.mSurfaceValue, page.mHostAccess.mTransport.mValue.mMetronomeActive).setTypeToggle()

    surfaceElements.transport.btnPan.mSurfaceValue.mOnProcessValueChange = function(activeDevice, newValue) 
    {
        if (newValue == 0) return
        changePageKeyPressed(activeDevice, PAGES.PAN)
	}

    surfaceElements.transport.btnEQ.mSurfaceValue.mOnProcessValueChange = function(activeDevice, newValue) 
    {
        if (newValue == 0) return
        changePageKeyPressed(activeDevice, PAGES.EQ)
	}

    surfaceElements.transport.btnFX.mSurfaceValue.mOnProcessValueChange = function(activeDevice, newValue) 
    {
        if (newValue == 0) return
        changePageKeyPressed(activeDevice, PAGES.FX)
	}

    surfaceElements.transport.btnSend.mSurfaceValue.mOnProcessValueChange = function(activeDevice, newValue) 
    {
        if (newValue == 0) return
        changePageKeyPressed(activeDevice, PAGES.SEND0)
	}


    // # FX / Quick controls

    var selectedChannel = page.mHostAccess.mTrackSelection.mMixerChannel;

    for(var i = 0; i < 8; i++)
    {
        var qcSlot = selectedChannel.mQuickControls.getByIndex(i)
        page.makeValueBinding(surfaceElements.faderStrips[i].knob.mSurfaceValue, qcSlot).setSubPage(subPageQuickCtrls)
    }


    var fxParamsViewer = selectedChannel.mInsertAndStripEffects.makeInsertEffectViewer("insertsViewer")        
    fxParamsViewer.followPluginWindowInFocus().excludeEmptySlots()

    
    // # EQ

    page.makeValueBinding(surfaceElements.virtuals.EQOns[0], selectedChannel.mChannelEQ.mBand1.mOn).setSubPage(subPageEQ)
    page.makeValueBinding(surfaceElements.faderStrips[0].knob.mSurfaceValue, selectedChannel.mChannelEQ.mBand1.mFilterType).setSubPage(subPageEQ)
    page.makeValueBinding(surfaceElements.faderStrips[1].knob.mSurfaceValue, selectedChannel.mChannelEQ.mBand1.mFreq).setSubPage(subPageEQ)
    page.makeValueBinding(surfaceElements.faderStrips[2].knob.mSurfaceValue, selectedChannel.mChannelEQ.mBand1.mGain).setSubPage(subPageEQ)
    page.makeValueBinding(surfaceElements.faderStrips[3].knob.mSurfaceValue, selectedChannel.mChannelEQ.mBand1.mQ).setSubPage(subPageEQ)
   
    page.makeValueBinding(surfaceElements.virtuals.EQOns[1], selectedChannel.mChannelEQ.mBand2.mOn).setSubPage(subPageEQ)
    page.makeValueBinding(surfaceElements.faderStrips[4].knob.mSurfaceValue, selectedChannel.mChannelEQ.mBand2.mFilterType).setSubPage(subPageEQ)
    page.makeValueBinding(surfaceElements.faderStrips[5].knob.mSurfaceValue, selectedChannel.mChannelEQ.mBand2.mFreq).setSubPage(subPageEQ)
    page.makeValueBinding(surfaceElements.faderStrips[6].knob.mSurfaceValue, selectedChannel.mChannelEQ.mBand2.mGain).setSubPage(subPageEQ)
    page.makeValueBinding(surfaceElements.faderStrips[7].knob.mSurfaceValue, selectedChannel.mChannelEQ.mBand2.mQ).setSubPage(subPageEQ)

    var secondEQPage = subPageEQ2
    var secondEQPageChanOffset = 0

    if (extensionCount > 1) // if 2 extensions then show one page on each
    {
        secondEQPage = subPageEQ
        secondEQPageChanOffset = 8
    }

    page.makeValueBinding(surfaceElements.virtuals.EQOns[2], selectedChannel.mChannelEQ.mBand3.mOn).setSubPage(secondEQPage)  
    page.makeValueBinding(surfaceElements.faderStrips[0 + secondEQPageChanOffset].knob.mSurfaceValue, selectedChannel.mChannelEQ.mBand3.mFilterType).setSubPage(secondEQPage)
    page.makeValueBinding(surfaceElements.faderStrips[1 + secondEQPageChanOffset].knob.mSurfaceValue, selectedChannel.mChannelEQ.mBand3.mFreq).setSubPage(secondEQPage)
    page.makeValueBinding(surfaceElements.faderStrips[2 + secondEQPageChanOffset].knob.mSurfaceValue, selectedChannel.mChannelEQ.mBand3.mGain).setSubPage(secondEQPage)
    page.makeValueBinding(surfaceElements.faderStrips[3 + secondEQPageChanOffset].knob.mSurfaceValue, selectedChannel.mChannelEQ.mBand3.mQ).setSubPage(secondEQPage)

    page.makeValueBinding(surfaceElements.virtuals.EQOns[3], selectedChannel.mChannelEQ.mBand4.mOn).setSubPage(secondEQPage)
    page.makeValueBinding(surfaceElements.faderStrips[4 + secondEQPageChanOffset].knob.mSurfaceValue, selectedChannel.mChannelEQ.mBand4.mFilterType).setSubPage(secondEQPage)
    page.makeValueBinding(surfaceElements.faderStrips[5 + secondEQPageChanOffset].knob.mSurfaceValue, selectedChannel.mChannelEQ.mBand4.mFreq).setSubPage(secondEQPage)
    page.makeValueBinding(surfaceElements.faderStrips[6 + secondEQPageChanOffset].knob.mSurfaceValue, selectedChannel.mChannelEQ.mBand4.mGain).setSubPage(secondEQPage)
    page.makeValueBinding(surfaceElements.faderStrips[7 + secondEQPageChanOffset].knob.mSurfaceValue, selectedChannel.mChannelEQ.mBand4.mQ).setSubPage(secondEQPage)

    // Write Trackname in top row if EQ or FX
    page.makeValueBinding(surfaceElements.virtuals.SelectedTitle, selectedChannel.mValue.mVolume)

    selectedChannel.mValue.mVolume.mOnTitleChange=(function(activeDevice, activeMapping, title)
    {
        selectedTrackTitle = title;

        if (activePage == PAGES.EQ || activePage == PAGES.EQ2 || activePage == PAGES.FX)
            writeText(activeDevice, 0, 0, selectedTrackTitle);        
    }).bind({globalTrackNo, trackNo, exID})


    // # Master

    var masterFader = page.mHostAccess.mMixConsole.makeMixerBankZone().includeOutputChannels().makeMixerBankChannel()
    // or to control room as alternative: mHostAccess.mControlRoom.mMainChannel.mLevelValue

    page.makeValueBinding(surfaceElements.transport.knob.mSurfaceValue, masterFader.mValue.mVolume)

    page.makeValueBinding(surfaceElements.transport.knobClick.mSurfaceValue, masterFader.mValue.mMute).setTypeToggle()
    makeButtonDisplayFeedback(surfaceElements.transport.knobClick, 0, 56)


    // # Strips
    
    var hostMixerBankZone = page.mHostAccess.mMixConsole.makeMixerBankZone()
        .excludeInputChannels()
        .excludeOutputChannels()
        .setFollowVisibility(true)

    if (!displayMidiTracks)
        hostMixerBankZone = hostMixerBankZone.excludeMIDIChannels()

    page.makeActionBinding(surfaceElements.transport.btnBankLeft.mSurfaceValue, hostMixerBankZone.mAction.mPrevBank)
    page.makeActionBinding(surfaceElements.transport.btnBankRight.mSurfaceValue, hostMixerBankZone.mAction.mNextBank)

    page.makeActionBinding(surfaceElements.transport.btnBankLeftDbl.mSurfaceValue, hostMixerBankZone.mAction.mShiftLeft)
    page.makeActionBinding(surfaceElements.transport.btnBankRightDbl.mSurfaceValue, hostMixerBankZone.mAction.mShiftRight)

    for(var exID = 0; exID < extensionCount; exID++) 
    {
        for(var trackNo = 0; trackNo < 8; trackNo++)
        {
            var globalTrackNo = exID * 8 + trackNo;

            var channelBankItem = hostMixerBankZone.makeMixerBankChannel()

            page.makeValueBinding(surfaceElements.faderStrips[globalTrackNo].btnMute.mSurfaceValue, channelBankItem.mValue.mMute).setTypeToggle()
            page.makeValueBinding(surfaceElements.faderStrips[globalTrackNo].btnSolo.mSurfaceValue, channelBankItem.mValue.mSolo).setTypeToggle()
            page.makeValueBinding(surfaceElements.faderStrips[globalTrackNo].btnRec.mSurfaceValue, channelBankItem.mValue.mRecordEnable).setTypeToggle()
            page.makeValueBinding(surfaceElements.faderStrips[globalTrackNo].btnSelect.mSurfaceValue, channelBankItem.mValue.mSelected).setTypeToggle()
            page.makeValueBinding(surfaceElements.faderStrips[globalTrackNo].btnSelectDbl.mSurfaceValue, channelBankItem.mValue.mInstrumentOpen).setTypeToggle()

            page.makeValueBinding(surfaceElements.faderStrips[globalTrackNo].fader.mSurfaceValue, channelBankItem.mValue.mVolume).setSubPage(subPageFaderVolume)

            if (selectTrackWhenTouched)
                page.makeValueBinding(surfaceElements.faderStrips[globalTrackNo].btnFaderTouch.mSurfaceValue, channelBankItem.mValue.mSelected).setTypeToggle()

            makeButtonDisplayFeedback(surfaceElements.faderStrips[globalTrackNo].btnMute, exID, trackNo + 16)
            makeButtonDisplayFeedback(surfaceElements.faderStrips[globalTrackNo].btnSolo, exID, trackNo + 8)
            makeButtonDisplayFeedback(surfaceElements.faderStrips[globalTrackNo].btnRec, exID, trackNo + 0)
            makeButtonDisplayFeedback(surfaceElements.faderStrips[globalTrackNo].btnSelect, exID, trackNo + 24)


            // Button commands, double clicks
            page.makeCommandBinding(surfaceElements.faderStrips[globalTrackNo].btnMuteDbl.mSurfaceValue, 'Edit', 'Unmute All')
            page.makeCommandBinding(surfaceElements.faderStrips[globalTrackNo].btnSoloDbl.mSurfaceValue, 'Edit', 'Deactivate All Solo')


            // # Encoders & Rings

            surfaceElements.faderStrips[globalTrackNo].knob.mSurfaceValue.mOnProcessValueChange=(function(activeDevice, value)
            {
                if (widerPanRing)
                    setLedRing(activeDevice, this.globalTrackNo, value, 7)
                else
                    setLedRing(activeDevice, this.globalTrackNo, value, 2)
            }).bind({globalTrackNo, trackNo, exID})

            surfaceElements.faderStrips[globalTrackNo].knob.mSurfaceValue.mOnTitleChange=(function(activeDevice, activeMapping, title) // todoooooooooooooooo
            {           
                if (activePage == PAGES.FX || activePage == PAGES.QUICKCTRLS)
                {
                    writeText(activeDevice, this.globalTrackNo, 2, ShortenText(title, 8));
                }                
            }).bind({globalTrackNo, trackNo, exID})

            surfaceElements.faderStrips[globalTrackNo].knob.mSurfaceValue.mOnDisplayValueChange=(function(activeDevice, text)
            {               
                surfaceElements.faderStrips[this.globalTrackNo].displayLineLowerTextPage = text;

                if (activePage == PAGES.TRACK)
                    return

                writeText(activeDevice, this.globalTrackNo, 1, text);
            }).bind({globalTrackNo, trackNo, exID})

            // Encoder Click
            surfaceElements.faderStrips[globalTrackNo].knobClick.mSurfaceValue.mOnProcessValueChange = function(activeDevice, newValue) 
            {
                if (newValue != 1) return

                if (activePage == PAGES.PAN || activePage == PAGES.TRACK)
                {
                    surfaceElements.faderStrips[this.globalTrackNo].vPanSettable.setProcessValue(activeDevice, 0.5)
                }
                else if (activePage == PAGES.EQ)
                {
                    ToggleValue(activeDevice, surfaceElements.virtuals.EQOns[Math.floor(this.globalTrackNo / 4)])
                }
                else if (activePage == PAGES.EQ2)
                {
                    ToggleValue(activeDevice, surfaceElements.virtuals.EQOns[Math.floor(this.globalTrackNo / 4) + 2])
                }
                else if (activePage >= PAGES.SEND0)
                {
                    ToggleValue(activeDevice, surfaceElements.faderStrips[this.globalTrackNo].vSendsOns[activePage - PAGES.SEND0])
                }
            }.bind({globalTrackNo})


            // # Pan

            page.makeValueBinding(surfaceElements.faderStrips[globalTrackNo].knob.mSurfaceValue, channelBankItem.mValue.mPan).setSubPage(subPagePan)
            page.makeValueBinding(surfaceElements.faderStrips[globalTrackNo].knob.mSurfaceValue, channelBankItem.mValue.mPan).setSubPage(subPageVolume)
            page.makeValueBinding(surfaceElements.faderStrips[globalTrackNo].vPanSettable, channelBankItem.mValue.mPan)


            // # Sends

            for(var i = 0; i < visibleSendTracks; i++)
            {
                var sendSlot = channelBankItem.mSends.getByIndex(i)
                page.makeValueBinding(surfaceElements.faderStrips[globalTrackNo].knob.mSurfaceValue, sendSlot.mLevel).setSubPage(subPageSends[i])
                page.makeValueBinding(surfaceElements.faderStrips[globalTrackNo].vSendsOns[i], sendSlot.mOn).setSubPage(subPageSends[i])

                page.makeValueBinding(surfaceElements.faderStrips[globalTrackNo].fader.mSurfaceValue, sendSlot.mLevel).setSubPage(subPageFaderSends[i])
            }


            // # FX

            var instrumentSlot = fxParamsViewer.mParameterBankZone.makeParameterValue()
            page.makeValueBinding(surfaceElements.faderStrips[globalTrackNo].knob.mSurfaceValue, instrumentSlot).setSubPage(subPageFX)


            // # VU           

            vuMeterLastSend[globalTrackNo] = [];
            vuMeterLastSend[globalTrackNo][0] = new Date().valueOf() - 5000;
            vuMeterLastSend[globalTrackNo][1] = new Date().valueOf() - 5000;

            page.makeValueBinding(surfaceElements.faderStrips[globalTrackNo].vVUMeter, channelBankItem.mValue.mVUMeter)

            channelBankItem.mValue.mVUMeter.mOnProcessValueChange=(function(activeDevice, activeMapping, value)
            {
                if (!vuEnabled) return

                var milliseconds = new Date().valueOf();

                // if (vuMeterLastSend[this.globalTrackNo][0] < milliseconds - 50) // limit updates to 5 FPS 200ms
                // {
                    displayValue = 60 + 9 * Math.log(value);

                    if (displayValue < 1)
                        displayValue = 0;

                    if (displayValue > 60)
                        displayValue = 60;

                    displayValue |= 0; // float to int
                    
                    sendMidiPolyKeyPress(activeDevice, this.exID, 0, this.trackNo, displayValue)
                    vuMeterLastSend[this.globalTrackNo][0] = milliseconds; // millisecs
                // }    
            }).bind({globalTrackNo, trackNo, exID})

            page.makeValueBinding(surfaceElements.faderStrips[globalTrackNo].vVUMeterClip, channelBankItem.mValue.mVUMeterClip)

            channelBankItem.mValue.mVUMeterClip.mOnProcessValueChange=(function(activeDevice, activeMapping, value)
            {
                // TODO
                if (!vuEnabled) return
                //console.log("Clipping: " + value + "    Track: " + this.globalTrackNo)
            }).bind({globalTrackNo, trackNo, exID})


            // # Faders

            surfaceElements.faderStrips[globalTrackNo].fader.mSurfaceValue.mOnProcessValueChange=(function(activeDevice, value)
            {
                sendMidiPitchBend(activeDevice, this.exID, this.trackNo, (value * 16383))
            }).bind({globalTrackNo, trackNo, exID})


            // # Text

            channelBankItem.mOnTitleChange=(function(activeDevice, activeMapping, title)
            {

                if (activePage == PAGES.EQ || activePage == PAGES.EQ2 || activePage == PAGES.FX || activePage == PAGES.QUICKCTRLS)
                    return

                writeText(activeDevice, this.globalTrackNo, 0, title);

            }).bind({globalTrackNo, trackNo, exID})

            surfaceElements.faderStrips[globalTrackNo].fader.mSurfaceValue.mOnDisplayValueChange=(function(activeDevice, text, units)
            {
                surfaceElements.faderStrips[this.globalTrackNo].displayLineLowerText = text;

                if (activePage == PAGES.TRACK || activePage >= PAGES.SEND0)
                    writeText(activeDevice, this.globalTrackNo, 1, text);
            }).bind({globalTrackNo, trackNo, exID})


            // # Colors

            channelBankItem.mOnColorChange=(function(activeDevice, activeMapping, r, g, b, a, isActive)
            {
                setColor(activeDevice, this.globalTrackNo, r, g, b);
            }).bind({globalTrackNo, trackNo, exID})           
        }
    }

    return page
}

function changePageKeyPressed(activeDevice, keyID)
{
    switch (keyID)
    {
        case PAGES.TRACK:
                activatePage(activeDevice, PAGES.TRACK);                       
            break;
        case PAGES.SEND0:
            if (activePage < PAGES.SEND0)
            {
                activatePage(activeDevice, PAGES.SEND0);
            }
            else
            {
                if (activePage == (PAGES.SEND0 + visibleSendTracks - 1))
                    activatePage(activeDevice, PAGES.TRACK);
                else
                    activatePage(activeDevice, activePage + 1);
            }
            break;
        case PAGES.PAN:
            if (activePage != PAGES.PAN)
                activatePage(activeDevice, PAGES.PAN);
            else
                activatePage(activeDevice, PAGES.TRACK);
            break;
        case PAGES.EQ: // EQ
            if (activePage != PAGES.EQ && activePage != PAGES.EQ2)
            {
                activatePage(activeDevice, PAGES.EQ);
            }
            else
            {        
                if (activePage == PAGES.EQ && extensionCount == 1)
                    activatePage(activeDevice, PAGES.EQ2); 
                else
                    activatePage(activeDevice, PAGES.TRACK); 
            }
            break;
        case PAGES.FX: // Plugin
            if (activePage != PAGES.FX && activePage != PAGES.QUICKCTRLS)
            {
                activatePage(activeDevice, PAGES.FX);
            }
            else
                if (activePage == PAGES.FX)
                    activatePage(activeDevice, PAGES.QUICKCTRLS);   
                else
                    activatePage(activeDevice, PAGES.TRACK);
            break;
    }
}

function activatePage(activeDevice, page)
{
    console.log("activatePage: "+ "    page: " + page)
    if (activePage == page)
        return;

    activePage = page;

    if (page == PAGES.TRACK)	
	{
        surfaceElements.virtuals.ActivatePageVolume.setProcessValue(activeDevice, 1)
        surfaceElements.virtuals.ActivatePageFaderVolume.setProcessValue(activeDevice, 1)
	}
	else if (page == PAGES.PAN)	
	{
        surfaceElements.virtuals.ActivatePagePan.setProcessValue(activeDevice, 1)
        surfaceElements.virtuals.ActivatePageFaderVolume.setProcessValue(activeDevice, 1)
	}
	else if (page == PAGES.FX)	
	{
        clearDisplay(activeDevice);
        surfaceElements.virtuals.ActivatePageFX.setProcessValue(activeDevice, 1)
        surfaceElements.virtuals.ActivatePageFaderVolume.setProcessValue(activeDevice, 1)
	}
    else if (page == PAGES.QUICKCTRLS)	
	{
        clearDisplay(activeDevice);
        surfaceElements.virtuals.ActivatePageQuickCtrls.setProcessValue(activeDevice, 1)
        surfaceElements.virtuals.ActivatePageFaderVolume.setProcessValue(activeDevice, 1)
	}
	else if (page == PAGES.EQ)	
	{
        clearDisplay(activeDevice);
        surfaceElements.virtuals.ActivatePageEQ.setProcessValue(activeDevice, 1)
        surfaceElements.virtuals.ActivatePageFaderVolume.setProcessValue(activeDevice, 1)
	}
    else if (page == PAGES.EQ2)	
	{
        surfaceElements.virtuals.ActivatePageEQ2.setProcessValue(activeDevice, 1)
        surfaceElements.virtuals.ActivatePageFaderVolume.setProcessValue(activeDevice, 1)
	}
	else if (page >= PAGES.SEND0)	
	{
        surfaceElements.virtuals.ActivatePageSends[page - PAGES.SEND0].setProcessValue(activeDevice, 1)

        if (showSendsOnFaders)
            surfaceElements.virtuals.ActivatePageFaderSends[page - PAGES.SEND0].setProcessValue(activeDevice, 1)
        else
            surfaceElements.virtuals.ActivatePageFaderVolume.setProcessValue(activeDevice, 1)
	}	
}

function showPage(activeDevice, page)
{
    console.log("showPage: "+ "    page: " + page)
    if (page == PAGES.TRACK)	
	{
		sendMidiNoteOn(activeDevice, 0, 0, VPOT_ASSIGN.EQ, 0);
		sendMidiNoteOn(activeDevice, 0, 0, VPOT_ASSIGN.SEND, 0);
		sendMidiNoteOn(activeDevice, 0, 0, VPOT_ASSIGN.PAN, 0);
		sendMidiNoteOn(activeDevice, 0, 0, VPOT_ASSIGN.FX, 0);
		setDisplayEncoderFunctionForPage(activeDevice, " ");
        clearTrackNos(activeDevice);

        for (var i = 0; i < totalTrackCount; i++) 
            writeText(activeDevice, i, 1, surfaceElements.faderStrips[i].displayLineLowerText)
	}
	else if (page == PAGES.PAN)	
	{
        console.log("activate pan: "+ "    page: " + page)
		sendMidiNoteOn(activeDevice, 0, 0, VPOT_ASSIGN.EQ, 0);
		sendMidiNoteOn(activeDevice, 0, 0, VPOT_ASSIGN.SEND, 0);
		sendMidiNoteOn(activeDevice, 0, 0, VPOT_ASSIGN.PAN, 127);
		sendMidiNoteOn(activeDevice, 0, 0, VPOT_ASSIGN.FX, 0);
		setDisplayEncoderFunctionForPage(activeDevice, "Pan");
        clearTrackNos(activeDevice);

        for (var i = 0; i < totalTrackCount; i++) 
            writeText(activeDevice, i, 1, surfaceElements.faderStrips[i].displayLineLowerTextPage)
	}
	else if (page == PAGES.FX)	
	{
		sendMidiNoteOn(activeDevice, 0, 0, VPOT_ASSIGN.EQ, 0);
		sendMidiNoteOn(activeDevice, 0, 0, VPOT_ASSIGN.SEND, 0);
		sendMidiNoteOn(activeDevice, 0, 0, VPOT_ASSIGN.PAN, 0);
		sendMidiNoteOn(activeDevice, 0, 0, VPOT_ASSIGN.FX, 127);

        writeText(activeDevice, 7, 0, "      FX QC");
        writeText(activeDevice, 0, 0, selectedTrackTitle);

        for (var i = 0; i < totalTrackCount; i++) 
        {
            showDisplayTrackNo(activeDevice, i, i + 1)            
        }       
	}
    else if (page == PAGES.QUICKCTRLS)	
	{
		sendMidiNoteOn(activeDevice, 0, 0, VPOT_ASSIGN.EQ, 0);
		sendMidiNoteOn(activeDevice, 0, 0, VPOT_ASSIGN.SEND, 0);
		sendMidiNoteOn(activeDevice, 0, 0, VPOT_ASSIGN.PAN, 0);
		sendMidiNoteOn(activeDevice, 0, 0, VPOT_ASSIGN.FX, 127);

        writeText(activeDevice, 7, 0, "Channel QC");
        writeText(activeDevice, 0, 0, selectedTrackTitle);

        for (var i = 0; i < 8; i++) 
        {
            showDisplayTrackNo(activeDevice, i, i + 1)            
        }       
	}
	else if (page == PAGES.EQ || page == PAGES.EQ2)	
	{
		sendMidiNoteOn(activeDevice, 0, 0, VPOT_ASSIGN.EQ, 127);
		sendMidiNoteOn(activeDevice, 0, 0, VPOT_ASSIGN.SEND, 0);
		sendMidiNoteOn(activeDevice, 0, 0, VPOT_ASSIGN.PAN, 0);
		sendMidiNoteOn(activeDevice, 0, 0, VPOT_ASSIGN.FX, 0);

        clearTrackNos(activeDevice);

        writeText(activeDevice, totalTrackCount - 1, 0, "          EQ");
        writeText(activeDevice, 7, 0, "          EQ");
        writeText(activeDevice, 0, 0, selectedTrackTitle);

        writeText(activeDevice, 0, 2, "1 Type");
        writeText(activeDevice, 1, 2, "1 Freq");
        writeText(activeDevice, 2, 2, "1 Gain");
        writeText(activeDevice, 3, 2, "1 Q");

        writeText(activeDevice, 4, 2, "2 Type");
        writeText(activeDevice, 5, 2, "2 Freq");
        writeText(activeDevice, 6, 2, "2 Gain");
        writeText(activeDevice, 7, 2, "2 Q");

        var secondEQPageChanOffset = 8

        if (page == PAGES.EQ2)
            secondEQPageChanOffset = 0

        if (extensionCount > 1 || page == PAGES.EQ2)
        {
            writeText(activeDevice, 0 + secondEQPageChanOffset, 2, "3 Type");
            writeText(activeDevice, 1 + secondEQPageChanOffset, 2, "3 Freq");
            writeText(activeDevice, 2 + secondEQPageChanOffset, 2, "3 Gain");
            writeText(activeDevice, 3 + secondEQPageChanOffset, 2, "3 Q");

            writeText(activeDevice, 4 + secondEQPageChanOffset, 2, "4 Type");
            writeText(activeDevice, 5 + secondEQPageChanOffset, 2, "4 Freq");
            writeText(activeDevice, 6 + secondEQPageChanOffset, 2, "4 Gain");
            writeText(activeDevice, 7 + secondEQPageChanOffset, 2, "4 Q");
        }
	}
	else if (page >= PAGES.SEND0)	
	{
		sendMidiNoteOn(activeDevice, 0, 0, VPOT_ASSIGN.EQ, 0);
		sendMidiNoteOn(activeDevice, 0, 0, VPOT_ASSIGN.SEND, 127);
		sendMidiNoteOn(activeDevice, 0, 0, VPOT_ASSIGN.PAN, 0);
		sendMidiNoteOn(activeDevice, 0, 0, VPOT_ASSIGN.FX, 0);
		setDisplayEncoderFunctionForPage(activeDevice, "Send " + (page - PAGES.SEND0 + 1));
        clearTrackNos(activeDevice);
	}	
}

var pageMixer = makePageMixer()

pageMixer.mOnActivate = function(activeDevice) 
{
	console.log('D700 page "Mixer" activated')
}
