//
// Asparion 400
//
// www.asparion.de
// Copyright 2023
//
// v1.0

// TODO
//
// stereo metering -> not available yet

//-----------------------------------------------------------------------------
// INCLUDE common functions
//-----------------------------------------------------------------------------

var midiremote_api = require('midiremote_api_v1')


//-----------------------------------------------------------------------------
// USER SETTINGS - settings can be adjusted by user
//-----------------------------------------------------------------------------

var extensionCount = 1;

var selectTrackWhenTouched = true
var displayMidiTracks = false
var sendsFadersToZeroAtClose = true

// If no display is attached these can be set to false to improve performance
var vuEnabled = true
var hasDisplay = true


//-----------------------------------------------------------------------------
// VARS - global vars and consts
//-----------------------------------------------------------------------------

var DISPLAY_WIDTH = 64;
var SINGLE_DISPLAY_WIDTH = 8;

var Extensions = [];
var vuMeterLastSend = [];
var totalTrackCount = extensionCount * 8;

    
//-----------------------------------------------------------------------------
// DRIVER SETUP - create driver object, midi ports and detection information
//-----------------------------------------------------------------------------

var deviceDriver

if (extensionCount > 1)
    deviceDriver = midiremote_api.makeDeviceDriver("Asparion", "D400T & " +  (extensionCount - 1) + "x D400F", "www.asparion.de")
else
    deviceDriver = midiremote_api.makeDeviceDriver("Asparion", "D400T & 1x D400F ", "www.asparion.de")

var midiInput = {}
var midiOutput = {}
var detectionUnit = deviceDriver.makeDetectionUnit()

function initPorts(extensionCount)
{
    for (var i = 0; i < extensionCount; i++)
    {
        if (i == 0)
        {
            midiInput[i] = deviceDriver.mPorts.makeMidiInput('D400 Input')
            midiOutput[i] = deviceDriver.mPorts.makeMidiOutput('D400 Output')

            detectionUnit.detectPortPair(midiInput[i],  midiOutput[i])
                .expectInputNameEquals('D 400')
                .expectOutputNameEquals('D 400')
            
            detectionUnit.detectPortPair(midiInput[i],  midiOutput[i])
                .expectInputNameStartsWith('D 400')
                .expectInputNameEndsWith(' 1')
                .expectOutputNameStartsWith('D 400')
                .expectOutputNameEndsWith(' 1')
        }
        else
        {       
            midiInput[i] = deviceDriver.mPorts.makeMidiInput('D400F ' + i + '  Input')
            midiOutput[i] = deviceDriver.mPorts.makeMidiOutput('D400F ' + i + '  Output')

            detectionUnit.detectPortPair(midiInput[i],  midiOutput[i])
                .expectInputNameEquals('MIDIIN' + (i + 1) + ' (D 400)')
                .expectOutputNameEquals('MIDIOUT' + (i + 1) + ' (D 400)')

            detectionUnit.detectPortPair(midiInput[i],  midiOutput[i])      
                .expectInputNameStartsWith('D 400')
                .expectInputNameEndsWith(' ' + (i + 1))
                .expectOutputNameStartsWith('D 400')
                .expectOutputNameEndsWith(' ' + (i + 1))
        }
    }
}


//-----------------------------------------------------------------------------
// CLASSES
//-----------------------------------------------------------------------------

function DisplayStrip(extensionID)
{
    this.extensionID = extensionID;
    this.channelNos = [];
    this.clearAllText()
}

DisplayStrip.prototype.setDisplayText = function(trackNo, row, text)
{
    this.text[trackNo][row] = text;
};

DisplayStrip.prototype.clearAllText = function()
{
    this.text = [];

    for (var i = 0; i < 8; i++) // tracks
    {
        this.text[i] = [];

        for (var ii = 0; ii < 3; ii++) // rows
        {
            this.text[i][ii] = " ";
        }
    }
};

DisplayStrip.prototype.clearAllTextInRow = function(row)
{
    for (var i = 0; i < 8; i++) // tracks
    {
        this.text[i][row] = " ";
    }
};

DisplayStrip.prototype.setText = function(activeDevice, dispNo, row, text)
{
    if (dispNo > 8 || row > 2)
        return

    var len = SINGLE_DISPLAY_WIDTH

    if (row == 2)
        len = SINGLE_DISPLAY_WIDTH_THIRD

    var position = dispNo * len;

    text = text.slice(0, len)

    if (this.text[dispNo][row] == text)
        return

    this.text[dispNo][row] = text

    this.writeToOLED(activeDevice, row, position, text, len)
}

DisplayStrip.prototype.writeToOLED = function(activeDevice, row, position, text, len)
{
    var sysExMessage = [0xf0, 0x00, 0x00, 0x66, 0x14]; 

    text = text.slice(0, len)
    
    if (row == 1)
        position += DISPLAY_WIDTH;

    sysExMessage.push(0x18);
    sysExMessage.push(position); 

    for(var i = 0; i < len; ++i)
        sysExMessage.push(text.charCodeAt(i))
    
    sysExMessage.push(0xf7)

    if (hasDisplay)
        midiOutput[this.extensionID].sendMidi(activeDevice, sysExMessage)
}

function Extension(extensionID, activeDevice)
{
    this.extensionID = extensionID;
    this.displayStrip = new DisplayStrip(extensionID);
}

function ShortenText(text, maxChars)
{
    var dmyText = text.trim();

    if (dmyText.length > maxChars)
    {
        dmyText = dmyText.replace(/\s/g,'')
    }

    return dmyText
}


//-----------------------------------------------------------------------------
// FUNCS
//-----------------------------------------------------------------------------

function trackNoToExtensionID(trackNo)
{
    return (trackNo / 8) | 0;
}

function trackNoToTrackNo(trackNo)
{
    return trackNo % 8;
}

function writeText(activeDevice, trackNo, row, text)
{
    Extensions[trackNoToExtensionID(trackNo)].displayStrip.setText(activeDevice, trackNoToTrackNo(trackNo), row, text);
}

function resetDev(activeDevice) 
{
    
    var data = [0xf0, 0x00, 0x00, 0x66, 0x14, 0x62, 0xf7]
    midiOutput[0].sendMidi(activeDevice, data)

    if (sendsFadersToZeroAtClose)
    {
        data = [0xf0, 0x00, 0x00, 0x66, 0x14, 0x61, 0xf7]
        midiOutput[0].sendMidi(activeDevice, data)
    }

    console.log('D400 reset')
}

function clearDisplay(activeDevice) 
{
    for (var i = 0; i < extensionCount; i++)
    {
        Extensions[i].displayStrip.clearAllText()
    }

    var data = [0xf0, 0x00, 0x00, 0x66, 0x14, 0x42, 0xf7]
    midiOutput[0].sendMidi(activeDevice, data)
}

function clearDisplayRow(activeDevice, row) 
{
    if (row < 0 || row > 3)
        return;

    for (var i = 0; i < extensionCount; i++)
    {
        Extensions[i].displayStrip.clearAllTextInRow(row)
    }

    var data = [0xf0, 0x00, 0x00, 0x66, 0x14, 0x66 + row, 0xf7] // 66 to 69 for row 1 to 4
    midiOutput[0].sendMidi(activeDevice, data)
}

function sendMidiNoteOn(dev, channel, exID, ledID, val)
{   
    midiOutput[exID].sendMidi(dev, [ 0x90 + channel, ledID, val ])
}

function sendMidiChannelPressure(dev, exID, channel, val)
{
    midiOutput[exID].sendMidi(dev, [ 0xD0 + channel, val ])
}

function sendMidiPolyKeyPress(dev, exID, channel, ID, val)
{
    midiOutput[exID].sendMidi(dev, [ 0xA0 + channel, ID, val ])
}

function sendMidiPitchBend(dev, exID, channel, val)
{
    midiOutput[exID].sendMidi(dev, [ 0xe0 + channel, (val & 127), (val >> 7) & 127 ])
}

function sendMidiCC(dev, exID, channel, ID, val)
{
    midiOutput[exID].sendMidi(dev, [ 0xB0 + channel, ID, val ])
}

function makeButtonDisplayFeedback(button, exID, ledID) 
{
    button.mSurfaceValue.mOnProcessValueChange = function (activeDevice, newValue) 
    {
		sendMidiNoteOn(activeDevice, 0, this.exID, this.ledID, newValue)
	}.bind({exID, ledID})
}

function setLedRing(activeDevice, encoderCode, value, ledMode)
{
    var newVal = Math.round(value * 127);
    sendMidiCC(activeDevice, 0, ledMode, encoderCode, newVal);
}

function init()
{
    initPorts(extensionCount)
}

init()

deviceDriver.mOnDeactivate = function(activeDevice) 
{   
    resetDev(activeDevice)
    console.log('D400 deactivated')
}


deviceDriver.mOnActivate = function(activeDevice) 
{   
    console.log('D400 activated')
    resetDev(activeDevice) 

    for (var exNo = 0; exNo < extensionCount; exNo++)
    {
        Extensions[exNo] = new Extension(exNo, activeDevice);
    }

    values = [];
	ledMode = [];

    for ( var i = 0; i < 8; i++)
	{
		values[i] = -1;
		ledMode[i] = -1;
	}

    surfaceElements.transport.jog.mSurfaceValue.setProcessValue(activeDevice, 0.5) // set jog dmy knob line to center
}


//-----------------------------------------------------------------------------
// SURFACE LAYOUT - create control elements and midi bindings
//-----------------------------------------------------------------------------

var surface = deviceDriver.mSurface

function bindMidiCC(button, exID, num) 
{
    button.mSurfaceValue.mMidiBinding.setInputPort(midiInput[exID]).bindToControlChange(0, num)
}

function bindNoteOn(button, exID, num) 
{
    button.mSurfaceValue.mMidiBinding.setInputPort(midiInput[exID]).bindToNote(0, num)
}

function makeFaderStrip(exID, channelIndex, x, y) 
{
    var faderStrip = {}
    var posFromTheTop = y

    if (selectTrackWhenTouched)
    {     
        faderStrip.btnFaderTouch = surface.makeButton(x + 2 * channelIndex + 0.7, posFromTheTop - 1, 0.6, 0.6).setShapeCircle()
        bindNoteOn(faderStrip.btnFaderTouch, exID, 104 + channelIndex)
    }

    posFromTheTop += 0
    faderStrip.fader = surface.makeFader(x + 2 * channelIndex + 0.3, posFromTheTop, 1.4, 11).setTypeVertical()

    posFromTheTop += 9

    faderStrip.fader.mSurfaceValue.mMidiBinding.setInputPort(midiInput[exID]).bindToPitchBend(channelIndex)

    posFromTheTop += 2.5
    faderStrip.btnMute = surface.makeButton(x + 2 * channelIndex, posFromTheTop + 0.2, 1, 0.8)
    faderStrip.btnRec = surface.makeButton(x + 2 * channelIndex + 1, posFromTheTop + 0.2, 1, 0.8)
    faderStrip.btnSelect = surface.makeButton(x + 2 * channelIndex + 0.4, posFromTheTop + 1, 1.3, 0.9)

    faderStrip.btnMuteDbl = surface.makeButton(0, 0, 0.01, 0.01)
    faderStrip.btnRecDbl = surface.makeButton(0, 0, 0.01, 0.01)
    faderStrip.btnSelectDbl = surface.makeButton(0, 0, 0.01, 0.01)

    bindNoteOn(faderStrip.btnMute, exID, 16 + channelIndex)
    bindNoteOn(faderStrip.btnRec, exID, 0 + channelIndex)
    bindNoteOn(faderStrip.btnSelect, exID, 24 + channelIndex)

    bindNoteOn(faderStrip.btnMuteDbl, exID, 8 + channelIndex)
    bindNoteOn(faderStrip.btnRecDbl, exID, 32 + channelIndex)
    bindNoteOn(faderStrip.btnSelectDbl, exID, 96 + channelIndex)  
    
    faderStrip.vVUMeter = surface.makeCustomValueVariable("vVUMeter")
    faderStrip.vVUMeterClip = surface.makeCustomValueVariable("vVUMeterClip")

    faderStrip.displayLineLowerText = ""
    faderStrip.displayLineLowerTextPage = ""

    return faderStrip
}

function makeTransport(x, y) 
{
    var transport = {}
    var posFromTheTop = y

    posFromTheTop = 7.8
    transport.btnShift = surface.makeButton(x - 0.1, posFromTheTop, 1.3, 1)
    bindNoteOn(transport.btnShift, 0, 70)

    // transport.btnShiftDbl = surface.makeButton(0, 0, 0.01, 0.01)
    // bindNoteOn(transport.btnShiftDbl, 0, 55)

    posFromTheTop = 10
    transport.btnA1 = surface.makeButton(x - 0.1, posFromTheTop, 1.3, 1)
    bindNoteOn(transport.btnA1, 0, 116)

    // transport.btnA1Dbl = surface.makeButton(0, 0, 0.01, 0.01)
    // bindNoteOn(transport.btnA1Dbl, 0, 55)

    posFromTheTop += 1.2;
    transport.btnA2 = surface.makeButton(x - 0.1, posFromTheTop, 1.3, 1)
    bindNoteOn(transport.btnA2, 0, 117)

    // transport.btnA2Dbl = surface.makeButton(0, 0, 0.01, 0.01)
    // bindNoteOn(transport.btnA2Dbl, 0, 55)

    posFromTheTop = 13
    transport.btnMagic = surface.makeButton(x - 0.1, posFromTheTop, 1.3, 1)
    bindNoteOn(transport.btnMagic, 0, 54)

    // transport.btnMagicDbl = surface.makeButton(0, 0, 0.01, 0.01)
    // bindNoteOn(transport.btnMagicDbl, 0, 55)

    // EQ
    posFromTheTop = 0.2; 
    posFromLeft = -0.3;
    transport.knobEQ1Gain = surface.makeKnob(x + posFromLeft, posFromTheTop, 1.7, 1.7)
    transport.knobEQ1Gain.mSurfaceValue.mMidiBinding.setInputPort(midiInput[0]).bindToControlChange(0, 0x61).setTypeRelativeSignedBit()
    transport.knobEQ1Width = surface.makeKnob(x + posFromLeft, posFromTheTop, 1.7, 1.7)
    transport.knobEQ1Width.mSurfaceValue.mMidiBinding.setInputPort(midiInput[0]).bindToControlChange(0, 0x68).setTypeRelativeSignedBit()
    transport.knobEQ1 = surface.makeKnob(x + posFromLeft, posFromTheTop, 1.7, 1.7)
    transport.knobEQ1.mSurfaceValue.mMidiBinding.setInputPort(midiInput[0]).bindToControlChange(0, 0x60).setTypeRelativeSignedBit()
    transport.knobEQ1DblClick = surface.makeButton(x + posFromLeft - 0.1, posFromTheTop + 0.5, 0.4, 0.4).setShapeCircle()
    bindNoteOn(transport.knobEQ1DblClick, 0, 68)
    transport.knobEQ1Click = surface.makeButton(x + posFromLeft - 0.1, posFromTheTop, 0.7, 0.7).setShapeCircle()
    bindNoteOn(transport.knobEQ1Click, 0, 64)   

    posFromTheTop += 1.8;  
    transport.knobEQ2Gain = surface.makeKnob(x + posFromLeft, posFromTheTop, 1.7, 1.7)
    transport.knobEQ2Gain.mSurfaceValue.mMidiBinding.setInputPort(midiInput[0]).bindToControlChange(0, 0x63).setTypeRelativeSignedBit()
    transport.knobEQ2Width = surface.makeKnob(x + posFromLeft, posFromTheTop, 1.7, 1.7)
    transport.knobEQ2Width.mSurfaceValue.mMidiBinding.setInputPort(midiInput[0]).bindToControlChange(0, 0x69).setTypeRelativeSignedBit()   
    transport.knobEQ2 = surface.makeKnob(x + posFromLeft, posFromTheTop, 1.7, 1.7)
    transport.knobEQ2.mSurfaceValue.mMidiBinding.setInputPort(midiInput[0]).bindToControlChange(0, 0x62).setTypeRelativeSignedBit()
    transport.knobEQ2DblClick = surface.makeButton(x + posFromLeft - 0.1, posFromTheTop + 0.5, 0.4, 0.4).setShapeCircle()
    bindNoteOn(transport.knobEQ2DblClick, 0, 69)
    transport.knobEQ2Click = surface.makeButton(x + posFromLeft - 0.1, posFromTheTop, 0.7, 0.7).setShapeCircle()
    bindNoteOn(transport.knobEQ2Click, 0, 65)   

    posFromTheTop += 1.8;   
    transport.knobEQ3Gain = surface.makeKnob(x + posFromLeft, posFromTheTop, 1.7, 1.7)
    transport.knobEQ3Gain.mSurfaceValue.mMidiBinding.setInputPort(midiInput[0]).bindToControlChange(0, 0x65).setTypeRelativeSignedBit()
    transport.knobEQ3Width = surface.makeKnob(x + posFromLeft, posFromTheTop, 1.7, 1.7)
    transport.knobEQ3Width.mSurfaceValue.mMidiBinding.setInputPort(midiInput[0]).bindToControlChange(0, 0x6A).setTypeRelativeSignedBit()
    transport.knobEQ3 = surface.makeKnob(x + posFromLeft, posFromTheTop, 1.7, 1.7)
    transport.knobEQ3.mSurfaceValue.mMidiBinding.setInputPort(midiInput[0]).bindToControlChange(0, 0x64).setTypeRelativeSignedBit()
    transport.knobEQ3DblClick = surface.makeButton(x + posFromLeft - 0.1, posFromTheTop + 0.5, 0.4, 0.4).setShapeCircle()
    bindNoteOn(transport.knobEQ3DblClick, 0, 70)
    transport.knobEQ3Click = surface.makeButton(x + posFromLeft - 0.1, posFromTheTop, 0.7, 0.7).setShapeCircle()
    bindNoteOn(transport.knobEQ3Click, 0, 66)
    
    posFromTheTop += 1.8;   
    transport.knobEQ4Gain = surface.makeKnob(x + posFromLeft, posFromTheTop, 1.7, 1.7)
    transport.knobEQ4Gain.mSurfaceValue.mMidiBinding.setInputPort(midiInput[0]).bindToControlChange(0, 0x67).setTypeRelativeSignedBit()
    transport.knobEQ4Width = surface.makeKnob(x + posFromLeft, posFromTheTop, 1.7, 1.7)
    transport.knobEQ4Width.mSurfaceValue.mMidiBinding.setInputPort(midiInput[0]).bindToControlChange(0, 0x6B).setTypeRelativeSignedBit()
    transport.knobEQ4 = surface.makeKnob(x + posFromLeft, posFromTheTop, 1.7, 1.7)
    transport.knobEQ4.mSurfaceValue.mMidiBinding.setInputPort(midiInput[0]).bindToControlChange(0, 0x66).setTypeRelativeSignedBit()
    transport.knobEQ4DblClick = surface.makeButton(x + posFromLeft - 0.1, posFromTheTop + 0.5, 0.4, 0.4).setShapeCircle()
    bindNoteOn(transport.knobEQ4DblClick, 0, 71)
    transport.knobEQ4Click = surface.makeButton(x + posFromLeft - 0.1, posFromTheTop, 0.7, 0.7).setShapeCircle()
    bindNoteOn(transport.knobEQ4Click, 0, 67)   

    // Jog
    posFromTheTop = 1.4; 
    posFromLeft = 3;
    transport.jog = surface.makeKnob(x + posFromLeft + 0.7, posFromTheTop + 5.7, 4, 4) // Knob is just for optics, working via buttons
    transport.jog.mSurfaceValue.mMidiBinding.setInputPort(midiInput[0]).bindToControlChange(0, 0x3C).setTypeRelativeSignedBit()

    transport.btnJogLeft = surface.makeButton(0, 0, 0.01, 0.01)
    bindNoteOn(transport.btnJogLeft, 0, 0x3F)
    transport.btnJogRight = surface.makeButton(0, 0, 0.01, 0.01)
    bindNoteOn(transport.btnJogRight, 0, 0x3E)

    // Vol 
    transport.knobVol = surface.makeKnob(x + posFromLeft, posFromTheTop, 1.7, 1.7)
    transport.knobVol.mSurfaceValue.mMidiBinding.setInputPort(midiInput[0]).bindToControlChange(0, 0x70).setTypeRelativeSignedBit()
    transport.knobVolClick = surface.makeButton(x + posFromLeft - 0.1, posFromTheTop, 0.7, 0.7).setShapeCircle()
    bindNoteOn(transport.knobVolClick, 0, 44)

    transport.knobVolDbl = surface.makeKnob(0, 0, 0.01, 0.01)
    transport.knobVolDbl.mSurfaceValue.mMidiBinding.setInputPort(midiInput[0]).bindToControlChange(0, 0x71).setTypeRelativeSignedBit()
    transport.knobVolClickDbl = surface.makeButton(0, 0, 0.01, 0.01).setShapeCircle()
    bindNoteOn(transport.knobVolClickDbl, 0, 49)

    // Pan
    transport.knobPan = surface.makeKnob(x + posFromLeft + 0.1, posFromTheTop + 3, 1.7, 1.7)
    transport.knobPan.mSurfaceValue.mMidiBinding.setInputPort(midiInput[0]).bindToControlChange(0, 0x72).setTypeRelativeSignedBit()
    transport.knobPanClick = surface.makeButton(x + posFromLeft - 0.1, posFromTheTop + 3, 0.7, 0.7).setShapeCircle()
    bindNoteOn(transport.knobPanClick, 0, 45)

    transport.knobPanDbl = surface.makeKnob(0, 0, 0.01, 0.01)
    transport.knobPanDbl.mSurfaceValue.mMidiBinding.setInputPort(midiInput[0]).bindToControlChange(0, 0x73).setTypeRelativeSignedBit()
    transport.knobPanClickDbl = surface.makeButton(0, 0, 0.01, 0.01).setShapeCircle()
    bindNoteOn(transport.knobPanClickDbl, 0, 50)

    transport.vPanSettable = surface.makeCustomValueVariable("vPanSettable")

    // Assignable ring
    posFromLeft += 3.5;
    transport.knobNorth = surface.makeKnob(x + posFromLeft + 0.1, posFromTheTop, 1.7, 1.7)
    transport.knobNorth.mSurfaceValue.mMidiBinding.setInputPort(midiInput[0]).bindToControlChange(0, 0x50).setTypeRelativeSignedBit()
    transport.knobNorthClick = surface.makeButton(x + posFromLeft - 0.1, posFromTheTop, 0.7, 0.7).setShapeCircle()
    bindNoteOn(transport.knobNorthClick, 0, 56)

    transport.knobNorthDbl = surface.makeKnob(0, 0, 0.01, 0.01)
    transport.knobNorthDbl.mSurfaceValue.mMidiBinding.setInputPort(midiInput[0]).bindToControlChange(0, 0x51).setTypeRelativeSignedBit()
    transport.knobNorthClickDbl = surface.makeButton(0, 0, 0.01, 0.01).setShapeCircle()
    bindNoteOn(transport.knobNorthClickDbl, 0, 57)

    transport.knobNorth3 = surface.makeKnob(0, 0, 0.01, 0.01)
    transport.knobNorth3.mSurfaceValue.mMidiBinding.setInputPort(midiInput[0]).bindToControlChange(0, 0x52).setTypeRelativeSignedBit()

    transport.knobNorth4 = surface.makeKnob(0, 0, 0.01, 0.01)
    transport.knobNorth4.mSurfaceValue.mMidiBinding.setInputPort(midiInput[0]).bindToControlChange(0, 0x53).setTypeRelativeSignedBit()

    transport.knobSouth = surface.makeKnob(x + posFromLeft, posFromTheTop + 3, 1.7, 1.7)
    transport.knobSouth.mSurfaceValue.mMidiBinding.setInputPort(midiInput[0]).bindToControlChange(0, 0x22).setTypeRelativeSignedBit()
    transport.knobSouthClick = surface.makeButton(x + posFromLeft - 0.1, posFromTheTop + 3, 0.7, 0.7).setShapeCircle()
    bindNoteOn(transport.knobSouthClick, 0, 56)

    transport.knobSouthDbl = surface.makeKnob(0, 0, 0.01, 0.01)
    transport.knobSouthDbl.mSurfaceValue.mMidiBinding.setInputPort(midiInput[0]).bindToControlChange(0, 0x23).setTypeRelativeSignedBit()
    transport.knobSouthClickDbl = surface.makeButton(0, 0, 0.01, 0.01).setShapeCircle()
    bindNoteOn(transport.knobSouthClickDbl, 0, 57)

    transport.knobSouth3 = surface.makeKnob(0, 0, 0.01, 0.01)
    transport.knobSouth3.mSurfaceValue.mMidiBinding.setInputPort(midiInput[0]).bindToControlChange(0, 0x24).setTypeRelativeSignedBit()

    transport.knobSouth4 = surface.makeKnob(0, 0, 0.01, 0.01)
    transport.knobSouth4.mSurfaceValue.mMidiBinding.setInputPort(midiInput[0]).bindToControlChange(0, 0x25).setTypeRelativeSignedBit()

    // 1 to 8
    posFromTheTop = 0.18; 
    posFromLeft = 9;
    transport.btn1 = surface.makeButton(x + posFromLeft, posFromTheTop + 0.2, 1, 0.68)
    bindNoteOn(transport.btn1, 0, 120)
    posFromTheTop += 0.75;
    transport.btn2 = surface.makeButton(x + posFromLeft, posFromTheTop + 0.2, 1, 0.68)
    bindNoteOn(transport.btn2, 0, 121)
    posFromTheTop += 0.75;
    transport.btn3 = surface.makeButton(x + posFromLeft, posFromTheTop + 0.2, 1, 0.68)
    bindNoteOn(transport.btn3, 0, 122)
    posFromTheTop += 0.75;
    transport.btn4 = surface.makeButton(x + posFromLeft, posFromTheTop + 0.2, 1, 0.68)
    bindNoteOn(transport.btn4, 0, 123)
    posFromTheTop += 1.1;
    transport.btn5 = surface.makeButton(x + posFromLeft, posFromTheTop + 0.2, 1, 0.68)
    bindNoteOn(transport.btn5, 0, 124)
    posFromTheTop += 0.75;
    transport.btn6 = surface.makeButton(x + posFromLeft, posFromTheTop + 0.2, 1, 0.68)
    bindNoteOn(transport.btn6, 0, 125)
    posFromTheTop += 0.75;
    transport.btn7 = surface.makeButton(x + posFromLeft, posFromTheTop + 0.2, 1, 0.68)
    bindNoteOn(transport.btn7, 0, 126)
    posFromTheTop += 0.75;
    transport.btn8 = surface.makeButton(x + posFromLeft, posFromTheTop + 0.2, 1, 0.68)
    bindNoteOn(transport.btn8, 0, 127)

    posFromTheTop = 9.2;
    transport.btnMetronome = surface.makeButton(x + 9, posFromTheTop, 1.3, 0.9)
    bindNoteOn(transport.btnMetronome, 0, 89)

    transport.btnCycle = surface.makeButton(x + 1.4, posFromTheTop, 1.3, 0.9)
    bindNoteOn(transport.btnCycle, 0, 86)

    transport.btnMetronomeDbl = surface.makeButton(0, 0, 0.01, 0.01)
    bindNoteOn(transport.btnMetronomeDbl, 0, 87)

    transport.btnCycleDbl = surface.makeButton(0, 0, 0.01, 0.01)
    bindNoteOn(transport.btnCycleDbl, 0, 82)


    posFromTheTop = 11;
    transport.btnSkipFwd = surface.makeButton(x + 8.1, posFromTheTop, 1.3, 0.9)
    bindNoteOn(transport.btnSkipFwd, 0, 92)

    transport.btnSkipBack = surface.makeButton(x + 2.1, posFromTheTop, 1.3, 0.9)
    bindNoteOn(transport.btnSkipBack, 0, 91)


    posFromTheTop = 12.7;
    transport.btnRecord = surface.makeButton(x + 7.3, posFromTheTop, 1.9, 1.3)
    bindNoteOn(transport.btnRecord, 0, 95)

    transport.btnStart = surface.makeButton(x + 5, posFromTheTop, 1.9, 1.3)
    bindNoteOn(transport.btnStart, 0, 94)

    transport.btnStop = surface.makeButton(x + 2.7, posFromTheTop, 1.9, 1.3)
    bindNoteOn(transport.btnStop, 0, 93)

    transport.btnRecordDbl = surface.makeButton(0, 0, 0.01, 0.01)
    bindNoteOn(transport.btnRecordDbl, 0, 85)

    transport.btnStartDbl = surface.makeButton(0, 0, 0.01, 0.01)
    bindNoteOn(transport.btnStartDbl, 0, 84)

    transport.btnStopDbl = surface.makeButton(0, 0, 0.01, 0.01)
    bindNoteOn(transport.btnStopDbl, 0, 83)


    posFromTheTop = 11.4;  
    transport.btnBankLeft = surface.makeButton(x + 9.9, posFromTheTop, 0.9, 1.3)
    bindNoteOn(transport.btnBankLeft, 0, 46)

    posFromTheTop += 1.4
    transport.btnBankRight = surface.makeButton(x + 9.9, posFromTheTop, 0.9, 1.3)
    bindNoteOn(transport.btnBankRight, 0, 47)


    transport.btnBankLeftDbl = surface.makeButton(0, 0, 0.01, 0.01)
    bindNoteOn(transport.btnBankLeftDbl, 0, 48)

    transport.btnBankRightDbl = surface.makeButton(0, 0, 0.01, 0.01)
    bindNoteOn(transport.btnBankRightDbl, 0, 49)

    return transport
}

function makeSurface() 
{
    var surfaceElements = {}

    surfaceElements.numStrips = extensionCount * 8
    surfaceElements.faderStrips = {}

    for(var exID = 0; exID < extensionCount; ++exID) 
    {
        for(var trackNo = 0; trackNo < 8; ++trackNo)
        {
            var globalTrackNo = exID * 8 + trackNo;
            surfaceElements.faderStrips[globalTrackNo] = makeFaderStrip(exID, trackNo, 0.4 + exID * 18, 1)
        }
    }

    surfaceElements.transport = makeTransport(extensionCount * 18, 0)
    
    return surfaceElements
}

function ToggleValue(activeDevice, surfaceEle)
{
    var isOn = surfaceEle.getProcessValue(activeDevice) == 1
    surfaceEle.setProcessValue(activeDevice, isOn ? 0 : 1)
}

var surfaceElements = makeSurface()

makeButtonDisplayFeedback(surfaceElements.transport.btnBankLeft, 0, 46)
makeButtonDisplayFeedback(surfaceElements.transport.btnBankRight, 0, 47)
makeButtonDisplayFeedback(surfaceElements.transport.btnStop, 0, 93)
makeButtonDisplayFeedback(surfaceElements.transport.btnStart, 0, 94)
makeButtonDisplayFeedback(surfaceElements.transport.btnRecord, 0, 95)

makeButtonDisplayFeedback(surfaceElements.transport.btnCycle, 0, 86)
makeButtonDisplayFeedback(surfaceElements.transport.btnMetronome, 0, 89)
makeButtonDisplayFeedback(surfaceElements.transport.btnMagic, 0, 54)

makeButtonDisplayFeedback(surfaceElements.transport.btnA1, 0, 116)
makeButtonDisplayFeedback(surfaceElements.transport.btnA2, 0, 117)

makeButtonDisplayFeedback(surfaceElements.transport.btnSkipBack, 0, 91)
makeButtonDisplayFeedback(surfaceElements.transport.btnSkipFwd, 0, 92)

makeButtonDisplayFeedback(surfaceElements.transport.btn1, 0, 120)
makeButtonDisplayFeedback(surfaceElements.transport.btn2, 0, 121)
makeButtonDisplayFeedback(surfaceElements.transport.btn3, 0, 122)
makeButtonDisplayFeedback(surfaceElements.transport.btn4, 0, 123)
makeButtonDisplayFeedback(surfaceElements.transport.btn5, 0, 124)
makeButtonDisplayFeedback(surfaceElements.transport.btn6, 0, 125)
makeButtonDisplayFeedback(surfaceElements.transport.btn7, 0, 126)
makeButtonDisplayFeedback(surfaceElements.transport.btn8, 0, 127)


//-----------------------------------------------------------------------------
// CREATE PAGES
//-----------------------------------------------------------------------------


var page = deviceDriver.mMapping.makePage("Mixer")


function makePageMixer() 
{

    page.makeValueBinding(surfaceElements.transport.btnStop.mSurfaceValue, page.mHostAccess.mTransport.mValue.mStop).setTypeToggle()
    page.makeValueBinding(surfaceElements.transport.btnStart.mSurfaceValue, page.mHostAccess.mTransport.mValue.mStart).setTypeToggle()    
    page.makeValueBinding(surfaceElements.transport.btnRecord.mSurfaceValue, page.mHostAccess.mTransport.mValue.mRecord).setTypeToggle()

    page.makeValueBinding(surfaceElements.transport.btnCycle.mSurfaceValue, page.mHostAccess.mTransport.mValue.mCycleActive).setTypeToggle()
    page.makeValueBinding(surfaceElements.transport.btnMetronome.mSurfaceValue, page.mHostAccess.mTransport.mValue.mMetronomeActive).setTypeToggle()

    page.makeValueBinding(surfaceElements.transport.btnSkipBack.mSurfaceValue, page.mHostAccess.mTransport.mValue.mRewind).setTypeDefault()
    page.makeValueBinding(surfaceElements.transport.btnSkipFwd.mSurfaceValue, page.mHostAccess.mTransport.mValue.mForward).setTypeDefault()

    // Magic, A1, A2 can be set in Cubase
    page.makeCommandBinding(surfaceElements.transport.btnMagic.mSurfaceValue, "Devices", "Mixer")
    // page.makeCommandBinding(surfaceElements.transport.btnA1.mSurfaceValue, 'Automation', 'Toggle Write Enable Selected Tracks')
    // page.makeCommandBinding(surfaceElements.transport.btnA2.mSurfaceValue, 'Automation', 'Toggle Read Enable Selected Tracks') // or 'Toggle Read Enable All Tracks'
    page.makeCommandBinding(surfaceElements.transport.btnA1.mSurfaceValue, 'Edit', 'Unmute All')
    page.makeCommandBinding(surfaceElements.transport.btnA2.mSurfaceValue, 'Edit', 'Deactivate All Solo')


    // # FX / Quick controls

    var selectedChannel = page.mHostAccess.mTrackSelection.mMixerChannel;

    var qcSlot1 = selectedChannel.mQuickControls.getByIndex(0)
    page.makeValueBinding(surfaceElements.transport.knobNorth.mSurfaceValue, qcSlot1)

    var qcSlot2 = selectedChannel.mQuickControls.getByIndex(1)
    page.makeValueBinding(surfaceElements.transport.knobNorthDbl.mSurfaceValue, qcSlot2)

    var qcSlot3 = selectedChannel.mQuickControls.getByIndex(2)
    page.makeValueBinding(surfaceElements.transport.knobNorth3.mSurfaceValue, qcSlot3)

    var qcSlot4 = selectedChannel.mQuickControls.getByIndex(3)
    page.makeValueBinding(surfaceElements.transport.knobNorth4.mSurfaceValue, qcSlot4)

    var qcSlot5 = selectedChannel.mQuickControls.getByIndex(4)
    page.makeValueBinding(surfaceElements.transport.knobSouth.mSurfaceValue, qcSlot5)

    var qcSlot6 = selectedChannel.mQuickControls.getByIndex(5)
    page.makeValueBinding(surfaceElements.transport.knobSouthDbl.mSurfaceValue, qcSlot6)

    var qcSlot7 = selectedChannel.mQuickControls.getByIndex(6)
    page.makeValueBinding(surfaceElements.transport.knobSouth3.mSurfaceValue, qcSlot7)

    var qcSlot8 = selectedChannel.mQuickControls.getByIndex(7)
    page.makeValueBinding(surfaceElements.transport.knobSouth4.mSurfaceValue, qcSlot8)

    surfaceElements.transport.knobNorth.mSurfaceValue.mOnProcessValueChange=(function(activeDevice, value)
    {
        setLedRing(activeDevice, 0x50, value, 0)
    })

    surfaceElements.transport.knobNorthDbl.mSurfaceValue.mOnProcessValueChange=(function(activeDevice, value)
    {
        setLedRing(activeDevice, 0x51, value, 0)
    })

    surfaceElements.transport.knobNorth3.mSurfaceValue.mOnProcessValueChange=(function(activeDevice, value)
    {
        setLedRing(activeDevice, 0x52, value, 0)
    })

    surfaceElements.transport.knobNorth4.mSurfaceValue.mOnProcessValueChange=(function(activeDevice, value)
    {
        setLedRing(activeDevice, 0x53, value, 0)
    })

    surfaceElements.transport.knobSouth.mSurfaceValue.mOnProcessValueChange=(function(activeDevice, value)
    {
        setLedRing(activeDevice, 0x22, value, 0)
    })

    surfaceElements.transport.knobSouthDbl.mSurfaceValue.mOnProcessValueChange=(function(activeDevice, value)
    {
        setLedRing(activeDevice, 0x23, value, 0)
    })

    surfaceElements.transport.knobSouth3.mSurfaceValue.mOnProcessValueChange=(function(activeDevice, value)
    {
        setLedRing(activeDevice, 0x24, value, 0)
    })

    surfaceElements.transport.knobSouth4.mSurfaceValue.mOnProcessValueChange=(function(activeDevice, value)
    {
        setLedRing(activeDevice, 0x25, value, 0)
    })

    
    // # EQ

    page.makeValueBinding(surfaceElements.transport.knobEQ1Click.mSurfaceValue, selectedChannel.mChannelEQ.mBand1.mOn).setTypeToggle()
    page.makeValueBinding(surfaceElements.transport.knobEQ2Click.mSurfaceValue, selectedChannel.mChannelEQ.mBand2.mOn).setTypeToggle()
    page.makeValueBinding(surfaceElements.transport.knobEQ3Click.mSurfaceValue, selectedChannel.mChannelEQ.mBand3.mOn).setTypeToggle()
    page.makeValueBinding(surfaceElements.transport.knobEQ4Click.mSurfaceValue, selectedChannel.mChannelEQ.mBand4.mOn).setTypeToggle()

    page.makeValueBinding(surfaceElements.transport.knobEQ1DblClick.mSurfaceValue, selectedChannel.mChannelEQ.mBand1.mFilterType).setTypeToggle()
    page.makeValueBinding(surfaceElements.transport.knobEQ2DblClick.mSurfaceValue, selectedChannel.mChannelEQ.mBand2.mFilterType).setTypeToggle()
    page.makeValueBinding(surfaceElements.transport.knobEQ3DblClick.mSurfaceValue, selectedChannel.mChannelEQ.mBand3.mFilterType).setTypeToggle()
    page.makeValueBinding(surfaceElements.transport.knobEQ4DblClick.mSurfaceValue, selectedChannel.mChannelEQ.mBand4.mFilterType).setTypeToggle()

    page.makeValueBinding(surfaceElements.transport.knobEQ1.mSurfaceValue, selectedChannel.mChannelEQ.mBand1.mFreq)
    page.makeValueBinding(surfaceElements.transport.knobEQ1Gain.mSurfaceValue, selectedChannel.mChannelEQ.mBand1.mGain)
    page.makeValueBinding(surfaceElements.transport.knobEQ1Width.mSurfaceValue, selectedChannel.mChannelEQ.mBand1.mQ)

    page.makeValueBinding(surfaceElements.transport.knobEQ2.mSurfaceValue, selectedChannel.mChannelEQ.mBand2.mFreq)
    page.makeValueBinding(surfaceElements.transport.knobEQ2Gain.mSurfaceValue, selectedChannel.mChannelEQ.mBand2.mGain)
    page.makeValueBinding(surfaceElements.transport.knobEQ2Width.mSurfaceValue, selectedChannel.mChannelEQ.mBand2.mQ)

    page.makeValueBinding(surfaceElements.transport.knobEQ3.mSurfaceValue, selectedChannel.mChannelEQ.mBand3.mFreq)
    page.makeValueBinding(surfaceElements.transport.knobEQ3Gain.mSurfaceValue, selectedChannel.mChannelEQ.mBand3.mGain)
    page.makeValueBinding(surfaceElements.transport.knobEQ3Width.mSurfaceValue, selectedChannel.mChannelEQ.mBand3.mQ)

    page.makeValueBinding(surfaceElements.transport.knobEQ4.mSurfaceValue, selectedChannel.mChannelEQ.mBand4.mFreq)
    page.makeValueBinding(surfaceElements.transport.knobEQ4Gain.mSurfaceValue, selectedChannel.mChannelEQ.mBand4.mGain)
    page.makeValueBinding(surfaceElements.transport.knobEQ4Width.mSurfaceValue, selectedChannel.mChannelEQ.mBand4.mQ)


    // # Master

    var masterFader = page.mHostAccess.mMixConsole.makeMixerBankZone().includeOutputChannels().makeMixerBankChannel()
    // or to control room as alternative: mHostAccess.mControlRoom.mMainChannel.mLevelValue

    page.makeValueBinding(surfaceElements.transport.knobVol.mSurfaceValue, masterFader.mValue.mVolume)

    surfaceElements.transport.knobVol.mSurfaceValue.mOnProcessValueChange=(function(activeDevice, value)
    {
        setLedRing(activeDevice, 0x70, value, 0)
    })

    page.makeValueBinding(surfaceElements.transport.knobVolClick.mSurfaceValue, masterFader.mValue.mMute).setTypeToggle()
    page.makeValueBinding(surfaceElements.transport.knobPan.mSurfaceValue, selectedChannel.mValue.mPan)

    surfaceElements.transport.knobPan.mSurfaceValue.mOnProcessValueChange=(function(activeDevice, value)
    {
        setLedRing(activeDevice, 0x72, value, 0)
    })

    page.makeValueBinding(surfaceElements.transport.knobPan.mSurfaceValue, selectedChannel.mValue.mPan)
    page.makeValueBinding(surfaceElements.transport.vPanSettable, selectedChannel.mValue.mPan)

    surfaceElements.transport.knobPanClick.mSurfaceValue.mOnProcessValueChange = function(activeDevice, newValue) 
    {
        if (newValue != 1) return

        surfaceElements.transport.vPanSettable.setProcessValue(activeDevice, 0.5)
    }.bind({globalTrackNo})


    // # Strips
    
    var hostMixerBankZone = page.mHostAccess.mMixConsole.makeMixerBankZone()
        .excludeInputChannels()
        .excludeOutputChannels()
        .setFollowVisibility(true)

    if (!displayMidiTracks)
        hostMixerBankZone = hostMixerBankZone.excludeMIDIChannels()

    page.makeActionBinding(surfaceElements.transport.btnBankLeft.mSurfaceValue, hostMixerBankZone.mAction.mPrevBank)
    page.makeActionBinding(surfaceElements.transport.btnBankRight.mSurfaceValue, hostMixerBankZone.mAction.mNextBank)

    page.makeActionBinding(surfaceElements.transport.btnBankLeftDbl.mSurfaceValue, hostMixerBankZone.mAction.mShiftLeft)
    page.makeActionBinding(surfaceElements.transport.btnBankRightDbl.mSurfaceValue, hostMixerBankZone.mAction.mShiftRight)

    page.makeCommandBinding(surfaceElements.transport.btnJogLeft.mSurfaceValue, 'Transport', 'Nudge Cursor Left') // 'Nudge Cursor Left', 'Jog Left') // jog version plays sound while scrolling
    page.makeCommandBinding(surfaceElements.transport.btnJogRight.mSurfaceValue, 'Transport', 'Nudge Cursor Right')

    for(var exID = 0; exID < extensionCount; exID++) 
    {
        for(var trackNo = 0; trackNo < 8; trackNo++)
        {
            var globalTrackNo = exID * 8 + trackNo;

            var channelBankItem = hostMixerBankZone.makeMixerBankChannel()

            page.makeValueBinding(surfaceElements.faderStrips[globalTrackNo].btnMute.mSurfaceValue, channelBankItem.mValue.mMute).setTypeToggle()
            page.makeValueBinding(surfaceElements.faderStrips[globalTrackNo].btnMuteDbl.mSurfaceValue, channelBankItem.mValue.mSolo).setTypeToggle()
            page.makeValueBinding(surfaceElements.faderStrips[globalTrackNo].btnRec.mSurfaceValue, channelBankItem.mValue.mRecordEnable).setTypeToggle()
            page.makeValueBinding(surfaceElements.faderStrips[globalTrackNo].btnRecDbl.mSurfaceValue, channelBankItem.mValue.mMonitorEnable).setTypeToggle()
            page.makeValueBinding(surfaceElements.faderStrips[globalTrackNo].btnSelect.mSurfaceValue, channelBankItem.mValue.mSelected).setTypeToggle()
            page.makeValueBinding(surfaceElements.faderStrips[globalTrackNo].btnSelectDbl.mSurfaceValue, channelBankItem.mValue.mInstrumentOpen).setTypeToggle()

            page.makeValueBinding(surfaceElements.faderStrips[globalTrackNo].fader.mSurfaceValue, channelBankItem.mValue.mVolume)

            if (selectTrackWhenTouched)
                page.makeValueBinding(surfaceElements.faderStrips[globalTrackNo].btnFaderTouch.mSurfaceValue, channelBankItem.mValue.mSelected).setTypeToggle()

            makeButtonDisplayFeedback(surfaceElements.faderStrips[globalTrackNo].btnMute, exID, trackNo + 16)
            makeButtonDisplayFeedback(surfaceElements.faderStrips[globalTrackNo].btnRec, exID, trackNo + 0)
            makeButtonDisplayFeedback(surfaceElements.faderStrips[globalTrackNo].btnSelect, exID, trackNo + 24)


            // # VU           

            vuMeterLastSend[globalTrackNo] = [];
            vuMeterLastSend[globalTrackNo][0] = new Date().valueOf() - 5000;
            vuMeterLastSend[globalTrackNo][1] = new Date().valueOf() - 5000;

            page.makeValueBinding(surfaceElements.faderStrips[globalTrackNo].vVUMeter, channelBankItem.mValue.mVUMeter)

            channelBankItem.mValue.mVUMeter.mOnProcessValueChange=(function(activeDevice, activeMapping, value)
            {
                if (!vuEnabled) return

                var milliseconds = new Date().valueOf();

                // if (vuMeterLastSend[this.globalTrackNo][0] < milliseconds - 50) // limit updates to 5 FPS 200ms
                // {
                    displayValue = 0xB + 2 * Math.log(value);

                    if (displayValue < 1)
                        displayValue = 0;

                    if (displayValue > 0xB)
                        displayValue = 0xB;

                    displayValue |= 0; // float to int
                    
                    sendMidiChannelPressure(activeDevice, this.exID, 0, (this.trackNo << 4) + displayValue)
                    vuMeterLastSend[this.globalTrackNo][0] = milliseconds; // millisecs
                // }    
            }).bind({globalTrackNo, trackNo, exID})

            page.makeValueBinding(surfaceElements.faderStrips[globalTrackNo].vVUMeterClip, channelBankItem.mValue.mVUMeterClip)

            channelBankItem.mValue.mVUMeterClip.mOnProcessValueChange=(function(activeDevice, activeMapping, value)
            {
                // TODO
                if (!vuEnabled) return
                //console.log("Clipping: " + value + "    Track: " + this.globalTrackNo)
            }).bind({globalTrackNo, trackNo, exID})


            // # Faders

            surfaceElements.faderStrips[globalTrackNo].fader.mSurfaceValue.mOnProcessValueChange=(function(activeDevice, value)
            {
                sendMidiPitchBend(activeDevice, this.exID, this.trackNo, (value * 16383))
            }).bind({globalTrackNo, trackNo, exID})


            // # Text

            channelBankItem.mOnTitleChange=(function(activeDevice, activeMapping, title)
            {
                writeText(activeDevice, this.globalTrackNo, 0, title);

            }).bind({globalTrackNo, trackNo, exID})

            surfaceElements.faderStrips[globalTrackNo].fader.mSurfaceValue.mOnDisplayValueChange=(function(activeDevice, text, units)
            {
                surfaceElements.faderStrips[this.globalTrackNo].displayLineLowerText = text;
                writeText(activeDevice, this.globalTrackNo, 1, text);
            }).bind({globalTrackNo, trackNo, exID})
        
        }
    }

    return page
}


var pageMixer = makePageMixer()

pageMixer.mOnActivate = function(activeDevice) 
{
	console.log('D400 page "Mixer" activated')
}
