import patterns
import mixer
import device
import transport
import arrangement
import general
import launchMapPages
import playlist
import ui
import channels

import midi
import utils
import time

# Consts

AsparionD400Page_Volume = 0
AsparionD400Page_Pan = 1
AsparionD400Page_Stereo = 2
AsparionD400Page_Channels = 3
AsparionD400Page_Sends = 4
AsparionD400Page_FX = 5 # on selected channel
AsparionD400Page_EQ = 6 # on selected channel

AsparionD400TotalPages = 3

AsparionD400STextCharactersPerRow = 64
AsparionD400STextCharactersPerDisplay = 8
AsparionD400STextRows = 2

MidiMsgNoteOnLightOn = 8323216 # 0x7F0090
MidiMsgNoteOnLightOff = 144 # 0x000090

II_Absolute = 0

#
# Settings
#
# Users can adjusted these settings to their liking, set "False" or "True"

SelectFirstTrackAtBankChange = False
ScrollToFirstTrackAtBankChange = False
SelectMultipleChannelsWithFaderTouch = False
SelectMultipleChannelsWithSelectButton = False


class TAsparionTrack:
    def __init__(self):
        self.TrackNum = 0
        self.BaseEventID = 0
        self.KnobEventID = 0
        self.KnobPressEventID = 0
        self.KnobResetEventID = 0
        self.KnobResetValue = 0
        self.KnobMode = 0
        self.KnobCenter = 0
        self.SliderEventID = 0
        self.Peak = 0
        self.Tag = 0
        self.SliderName = ""
        self.KnobName = ""
        self.LastValueIndex = 0
        self.ZPeak = False
        self.Dirty = False
        self.KnobHeld = False
        self.TextLine1 = ""
        self.TextLine2 = ""
        self.TextLine1NeedsUpdate = False
        self.TextLine2NeedsUpdate = False
        self.TextTemp = ""
        self.Name = ""
        self.TextValuePre = ""
        self.TextValue = ""
        self.Value = 0
        self.LastSentFaderPos = -1


class TAsparionD400:
    def __init__(self):
        self.TempMsgT = ["", ""]
        self.LastTimeMsg = bytearray(10)
        self.Shift = False
        self.TempMsgDirty = False
        self.JogSource = 0
        self.TempMsgCount = 0
        self.FirstTrack = 0
        self.FirstTrackT = [0, 0]
        self.Tracks = []
        self.Clicking = False
        self.Scrub = False
        self.Flip = False
        self.Page = AsparionD400Page_Volume
        self.SmoothSpeed = 0
        self.MeterMax = 0xD
        self.ActivityMax = 0xD
        self.ExNo = 0
        self.TrackOffset = 0
        self.IsMain = False
        self.SelectedTrackNo = 0
        self.SelectedTrackNoGlobal = 0
        self.AsparionD400_ExtenderPosT = ("left", "right")
        self.FreeEventID = 400
        self.AlphaTrack_SliderMax = round(13072 * 16000 / 12800)
        self.GlobalRingPositions = []
        self.GlobalRingEventIDs = []
        self.AdditionalReceiverCount = 0


    def OnInit(self, ExNo):

        if not device.isAssigned():
            return

        self.ExNo = ExNo
        self.IsMain = ExNo == 0
        self.TrackOffset = ExNo * 8 + 1
        self.FirstTrackT[0] = self.TrackOffset

        self.Tracks = [0 for x in range(9)]
        for x in range(0, 9):
            self.Tracks[x] = TAsparionTrack()

        self.GlobalRingPositions = [0 for x in range(9)]
        for x in range(0, 9):
            self.GlobalRingPositions[x] = 0

        self.GlobalRingEventIDs = [0 for x in range(9)]
        for x in range(0, 9):
            self.GlobalRingEventIDs[x] = 0
        
        self.FirstTrack = 0
        self.SmoothSpeed = 469
        self.Clicking = True

        self.ShowUpdateFirmware()

        device.setHasMeters()
        self.LastTimeMsg = bytearray(10)

        self.InitMeter()
        self.SetPage(self.Page)
        self.UpdateFader(8) # master vol

        self.PrintText("OnInit finished")


    def PrintText(self, text):

        print(time.strftime("%H:%M:%S") + ": " + text)


    def OnDeInit(self):

        if device.isAssigned():
            for m in range(0, 8):
                device.midiOutSysex(bytes([0xF0, 0x00, 0x00, 0x66, 0x14, 0x20, m, 0, 0xF7]))

            #self.ClearDispalyWithSpaces()

        self.TurnEverythingOff()

        self.PrintText("OnDeInit finished")


    def TriggerStartupScreen(self):
        if not self.IsMain:
            return

        sysex = bytearray([0xF0, 0x00, 0x00, 0x66, 0x14, 0x41])
        sysex.append(0xF7)
        device.midiOutSysex(bytes(sysex))


    def OnDirtyMixerTrack(self, SetTrackNum):

        for m in range(0, len(self.Tracks)):
            if (self.Tracks[m].TrackNum == SetTrackNum) | (SetTrackNum == -1):
                self.Tracks[m].Dirty = True


    def OnRefresh(self, flags):

        print("fu")
        if flags & midi.HW_Dirty_Mixer_Sel:
            self.UpdateSelectedTrack()
            
        if flags & midi.HW_Dirty_Mixer_Controls:
            self.UpdateSelectedTrack()
            

        if flags & midi.HW_Dirty_Mixer_Display or flags & midi.HW_Dirty_Mixer_Controls:
            self.UpdateTracks()
            

        # LEDs
        if flags & midi.HW_Dirty_LEDs:
            self.UpdateTransportLEDs()
            

        if flags & midi.HW_Dirty_RemoteLinks: 
            self.UpdateLinkedControllers()

        if  flags & midi.HW_Dirty_RemoteLinkValues:
            self.UpdateLinkedControllers()


    def Jog(self, event):

        transport.globalTransport(midi.FPT_Jog + int(self.Shift ^ self.Scrub), event.outEv, event.pmeFlags)  # relocate


    def OnMidiMsg(self, event):

        self.AdditionalReceiverCount = device.dispatchReceiverCount()

        if event.midiId == midi.MIDI_PROGRAMCHANGE:  
            if (self.IsMain):
                pass
            elif event.data1 == 0x44:
                self.SetPage(event.data2)
            elif event.data1 == 0x55:
                self.SetFirstTrack(self.FirstTrackT[self.FirstTrack] + event.data2)
            elif event.data1 == 0x56:
                self.SetFirstTrack(self.FirstTrackT[self.FirstTrack] - event.data2)
            
        elif event.midiId == midi.MIDI_CONTROLCHANGE:
            if event.midiChan == 0:
                event.inEv = event.data2
                if event.inEv >= 0x40:
                    event.outEv = -(event.inEv - 0x40)
                else:
                    event.outEv = event.inEv

                if event.data1 == 0x3C:
                    self.Jog(event)
                    event.handled = True

                if (event.data1 >= 0x50 and event.data1 <= 0x53) or (event.data1 >= 0x20 and event.data1 <= 0x25) or (event.data1 == 0x66): # A1 North Ring, A2 South Ring, EQ4
                    #pass
                    #device.processMIDICC(event)
                    eventID = device.findEventID(midi.EncodeRemoteControlID(device.getPortNumber(), 0, 0) + event.data1, 1)
                    mixer.automateEvent(eventID, event.outEv, midi.REC_Controller, self.SmoothSpeed, 1, midi.EKRes / 8)
                    self.UpdateLinkedControllers()
                    event.handled = True

                elif event.data1 >= 0x60 and event.data1 <= 0x6B:                  
                    currentEQID = mixer.getTrackPluginId(mixer.trackNumber(), 0)
                    event.handled = True

                    if event.data1 == 0x60:
                        currentEQID += midi.REC_Mixer_EQ_Freq
                    elif event.data1 == 0x61:
                        currentEQID += midi.REC_Mixer_EQ_Gain
                    elif event.data1 == 0x68:
                        currentEQID += midi.REC_Mixer_EQ_Q

                    elif event.data1 == 0x62:
                        currentEQID += midi.REC_Mixer_EQ_Freq + 1
                    elif event.data1 == 0x63:
                        currentEQID += midi.REC_Mixer_EQ_Gain + 1
                    elif event.data1 == 0x69:
                        currentEQID += midi.REC_Mixer_EQ_Q + 1

                    elif event.data1 == 0x64:
                        currentEQID += midi.REC_Mixer_EQ_Freq + 2
                    elif event.data1 == 0x65:
                        currentEQID += midi.REC_Mixer_EQ_Gain + 2
                    elif event.data1 == 0x6A:
                        currentEQID += midi.REC_Mixer_EQ_Q + 2

                    # Doesn't exist
                    # elif event.data1 == 0x66:
                    #     currentEQID += midi.REC_Mixer_EQ_Freq + 3
                    # elif event.data1 == 0x67:
                    #     currentEQID += midi.REC_Mixer_EQ_Gain + 3
                    # elif event.data1 == 0x6B:
                    #     currentEQID += midi.REC_Mixer_EQ_Q + 3

                    else:
                        event.handled = False

                    if event.handled:
                        mixer.automateEvent(currentEQID, event.outEv, midi.REC_Controller, self.SmoothSpeed, 1, midi.EKRes / 8)
                   

                elif event.data1 == 0x72: # pan encoder press
                    currentPanID = mixer.getTrackPluginId(mixer.trackNumber(), 0) + midi.REC_Mixer_Pan
                    mixer.automateEvent(currentPanID, event.outEv, midi.REC_Controller, self.SmoothSpeed, 1, midi.EKRes / 8)
                    event.handled = True

                elif event.data1 == 0x73: # pan encoder double press
                    currentSSID = mixer.getTrackPluginId(mixer.trackNumber(), 0) + midi.REC_Mixer_SS
                    mixer.automateEvent(currentSSID, event.outEv, midi.REC_Controller, self.SmoothSpeed, 1, midi.EKRes / 8)
                    event.handled = True

                else:
                    event.handled = False  # for extra CCs in emulators
            else:
                event.handled = False  # for extra CCs in emulators

        elif event.midiId == midi.MIDI_PITCHBEND:  # pitch bend (faders)

            
            if event.midiChan <= 8:
                event.inEv = event.data1 + (event.data2 << 7)
                event.outEv = (event.inEv << 16) // 16383
                event.inEv -= 0x2000
                
                if self.Tracks[event.midiChan].SliderEventID >= 0:
                    # slider (mixer track volume)
                    event.handled = True
                    mixer.automateEvent(
                        self.Tracks[event.midiChan].SliderEventID,
                        self.AlphaTrack_SliderToLevel(event.inEv + 0x2000),
                        midi.REC_MIDIController,
                        self.SmoothSpeed,
                    )                                  

        elif (event.midiId == midi.MIDI_NOTEON) | (event.midiId == midi.MIDI_NOTEOFF):  
            if event.midiId == midi.MIDI_NOTEON:

                if event.pmeFlags & midi.PME_System != 0:

                    event.handled = True

                    if (event.data1 == 0x2E) | (event.data1 == 0x2F):  # mixer bank, step 8
                        if event.data2 > 0:
                            incByTracks = (self.AdditionalReceiverCount + 1) * 8
                            #incByTracks = 8

                            self.SetFirstTrack(self.FirstTrackT[self.FirstTrack] + incByTracks * (1 if event.data1 == 0x2F else -1))

                            if self.IsMain:
                                for n in range(0, self.AdditionalReceiverCount):
                                    device.dispatch(n, midi.MIDI_PROGRAMCHANGE + ((0x55 if event.data1 == 0x2F else 0x56) << 8) + (incByTracks << 16))

                    elif (event.data1 == 0x30) | (event.data1 == 0x31): # channel bank, step 1
                        if event.data2 > 0:
                            incByTracks = 1
                            self.SetFirstTrack(self.FirstTrackT[self.FirstTrack] - 1 + int(event.data1 == 0x31) * 2)

                            if self.IsMain:
                                for n in range(0, self.AdditionalReceiverCount):
                                    device.dispatch(n, midi.MIDI_PROGRAMCHANGE + ((0x55 if event.data1 == 0x31 else 0x56) << 8) + (incByTracks << 16))

                    elif event.data1 in [0x68, 0x69, 0x6A, 0x6B, 0x6C, 0x6D, 0x6E, 0x6F]: # fader touch
                        if event.data2 > 0:
                            mixer.setTrackNumber(self.Tracks[event.data1- 0x68].TrackNum, midi.curfxMinimalLatencyUpdate | (midi.curfxNoDeselectAll if SelectMultipleChannelsWithFaderTouch else 0))

                    elif event.data1 == 0x32:  # Magic
                        if event.data2 > 0:
                            if self.IsMain:
                                self.SetNextPage()
                                self.UpdateTransportLEDs()

                    elif event.data1 == 0x65:  # self.Scrub
                        if event.data2 > 0:
                            self.Scrub = not self.Scrub
                            self.UpdateTransportLEDs()
          
                    elif event.data1 == 0x59:  # Metronome
                        if event.data2 > 0:
                            transport.globalTransport(midi.FPT_Metronome, 1, event.pmeFlags)

                    elif event.data1 == 0x58:  # Precount metronome, double?
                        if event.data2 > 0:
                            transport.globalTransport(midi.FPT_CountDown, 1, event.pmeFlags)

                    elif (event.data1 == 0x5B) | (event.data1 == 0x5C):  # << >>
                        if self.Shift:
                            if event.data2 == 0:
                                v2 = 1
                            elif event.data1 == 0x5B:
                                v2 = 0.5
                            else:
                                v2 = 2
                            transport.setPlaybackSpeed(v2)
                        else:
                            transport.globalTransport(
                                midi.FPT_Rewind + int(event.data1 == 0x5C),
                                int(event.data2 > 0) * 2,
                                event.pmeFlags,
                            )
                        device.directFeedback(event)

                    elif event.data1 == 0x5D:  # stop
                        transport.globalTransport(midi.FPT_Stop, int(event.data2 > 0) * 2, event.pmeFlags)
                    elif event.data1 == 0x5E:  # play
                        transport.globalTransport(midi.FPT_Play, int(event.data2 > 0) * 2, event.pmeFlags)
                    elif event.data1 == 0x5F:  # record
                        transport.globalTransport(midi.FPT_Record, int(event.data2 > 0) * 2, event.pmeFlags)

                    elif event.data1 == 0x56:  # song/loop changed from 0x5A to 0x56
                        transport.globalTransport(midi.FPT_Loop, int(event.data2 > 0) * 2, event.pmeFlags)

                    elif event.data1 == 0x52: # loop double
                        transport.globalTransport(midi.FPT_Snap, int(event.data2 > 0) * 2, event.pmeFlags)

                    elif event.pmeFlags & midi.PME_System_Safe != 0:

                        if (event.data1 >= 0x18) & (event.data1 <= 0x1F):  # select mixer track
                            if event.data2 > 0:
                                i = event.data1 - 0x18
                                ui.showWindow(midi.widMixer)
                                mixer.setTrackNumber(self.Tracks[i].TrackNum, midi.curfxScrollToMakeVisible | midi.curfxMinimalLatencyUpdate | (midi.curfxNoDeselectAll if SelectMultipleChannelsWithSelectButton else 0))

                        elif (event.data1 >= 0x8) & (event.data1 <= 0xF):  # solo
                            if event.data2 > 0:
                                i = event.data1 - 0x8
                                self.Tracks[i].solomode = midi.fxSoloModeWithDestTracks
                                mixer.soloTrack(self.Tracks[i].TrackNum, midi.fxSoloToggle, self.Tracks[i].solomode)
                                mixer.setTrackNumber(self.Tracks[i].TrackNum, midi.curfxScrollToMakeVisible)

                        elif (event.data1 >= 0x10) & (event.data1 <= 0x17):  # mute
                            if event.data2 > 0:
                                mixer.enableTrack(self.Tracks[event.data1 - 0x10].TrackNum)

                        elif (event.data1 >= 0x0) & (event.data1 <= 0x7):  # arm
                            if event.data2 > 0:
                                mixer.armTrack(self.Tracks[event.data1].TrackNum)
                        else:
                            event.handled = False
                    else:
                        event.handled = False


            elif event.midiId == midi.MIDI_NOTEOFF:

                event.handled = True 

                if event.data2 > 0:
                    event.handled = True
                    if event.data1 == 0x78:  # 1 open Playlist
                        transport.globalTransport(midi.FPT_F5, 1)
                    elif event.data1 == 0x79:  # 2 open Piano Roll
                        transport.globalTransport(midi.FPT_F7, 1)
                    elif event.data1 == 0x7A:  # 3 open Channel Rack
                        transport.globalTransport(midi.FPT_F6, 1)
                    elif event.data1 == 0x7B:  # 4 open Mixer
                        transport.globalTransport(midi.FPT_F9, 1)
                    elif event.data1 == 0x7C:  # 5 open View Browser, Alt + F8
                        if ui.getVisible(midi.widBrowser) == 1:
                            ui.hideWindow(midi.widBrowser)
                        elif ui.getVisible(midi.widBrowser) == 0:
                            ui.showWindow(midi.widBrowser)
                            ui.setFocused(midi.widBrowser)
                    elif event.data1 == 0x7D:  # 6 open Plugin Picker
                        transport.globalTransport(midi.FPT_F8, 1)
                    elif event.data1 == 0x7E:  # 7 open Tap Tempo
                        transport.globalTransport(midi.FPT_TapTempo, 1)
                    elif event.data1 == 0x7F:  # 8 Close all Windows
                        transport.globalTransport(midi.FPT_F12, 1)
                    elif event.data1 == 0x72:
                        mixer.setTrackPan(mixer.trackNumber(), 0)
                    elif event.data1 == 0x73:
                        currentSSID = mixer.getTrackPluginId(mixer.trackNumber(), 0) + midi.REC_Mixer_SS
                        mixer.automateEvent(currentSSID, 0, midi.REC_UpdateValue, II_Absolute)
                        mixer.automateEvent(currentSSID, 0, midi.REC_UpdatePlugLabel, II_Absolute)
                        mixer.automateEvent(currentSSID, 0, midi.REC_UpdateControl, II_Absolute)
                    elif event.data1 == 0x74:  # A1, add marker
                        if (transport.globalTransport(midi.FPT_AddMarker, 2, event.pmeFlags) == midi.GT_Global) & (event.data2 > 0):
                                self.OnSendTempMsg(ui.getHintMsg())
                    elif event.data1 == 0x75:  # A2, undo
                        general.undo()
                    else:
                        event.handled = False
            else:
                event.handled = False


    def WriteDisplayPerTrack(self, txt, Row, TrackNo):
        txt = txt.encode("ascii", "ignore").decode() # remove non ascii
        textLength = len(txt)

        txtEnding = ""

        if (textLength > 8):
            
            if txt[textLength - 3].isnumeric():
                txtEnding += txt[textLength - 3]
            if txt[textLength - 2].isnumeric():
                txtEnding += txt[textLength - 2]
            if txt[textLength - 1].isnumeric():
                txtEnding += txt[textLength - 1]

        txt = txt[:AsparionD400STextCharactersPerDisplay - len(txtEnding)]
        txt += txtEnding
        txt = txt.ljust(AsparionD400STextCharactersPerDisplay, " ")
        
        if (Row == 0):
            if (self.Tracks[TrackNo].TextLine1 == txt): # only update if changed
                return

            self.Tracks[TrackNo].TextLine1 = txt
        elif (Row == 1):
            if (self.Tracks[TrackNo].TextLine2 == txt): # only update if changed
                return

            self.Tracks[TrackNo].TextLine2 = txt
        else:
            return

        sysex = bytearray([0xF0, 0x00, 0x00, 0x66, 0x14, 0x18, AsparionD400STextCharactersPerRow * Row + TrackNo * AsparionD400STextCharactersPerDisplay]
            ) + bytearray(txt, "ascii")
        sysex.append(0xF7)
        device.midiOutSysex(bytes(sysex))


    def ShowUpdateFirmware(self):
        if self.IsMain == False:
            return

        sysex = bytearray([0xF0, 0x00, 0x00, 0x66, 0x14, 0x12, 0]
            ) + bytearray("Please update firmware      to 1.16or     higher        ", "ascii")
        sysex.append(0xF7)
        device.midiOutSysex(bytes(sysex))


    def TurnEverythingOff(self):
        if not device.isAssigned():
            return

        for no in range(0, 7):
            self.UpdateTactLED(0x00 + no, False)
            self.UpdateTactLED(0x08 + no, False)
            self.UpdateTactLED(0x10 + no, False)
            self.UpdateTactLED(0x18 + no, False)

        if self.IsMain:
            # stop
            self.UpdateTactLED(0x5D, False)
            # loop
            self.UpdateTactLED(0x56, False)
            # record
            self.UpdateTactLED(0x5F, False)
            # changed flag
            self.UpdateTactLED(0x50, False)
            # metronome
            self.UpdateTactLED(0x59, False)
            # rec precount
            self.UpdateTactLED(0x58, False)
            # smoothing
            self.UpdateTactLED(0x33, False)
            # Magic
            self.UpdateTactLED(0x32, False)

    	    # internal turn everything off
            sysex = bytearray([0xF0, 0x00, 0x00, 0x66, 0x14, 0x62])
            sysex.append(0xF7)
            device.midiOutSysex(bytes(sysex))


    def ClearDispalyWithSpaces(self):
        self.WriteDisplayDirty("", 0)
        self.WriteDisplayDirty("", 1)


    def WriteDisplayDirty(self, txt, Row):
        txt = txt.encode("ascii", "ignore").decode() # remove non ascii
        txt = txt[:AsparionD400STextCharactersPerRow].ljust(AsparionD400STextCharactersPerRow, " ")
        sysex = bytearray([0xF0, 0x00, 0x00, 0x66, 0x14, 0x18, AsparionD400STextCharactersPerRow * Row] 
            ) + bytearray(txt, "ascii")
        sysex.append(0xF7)
        device.midiOutSysex(bytes(sysex))


    def UpdateTempMsg(self):
        pass


    def OnSendTempMsg(self, Msg, Duration=1000):
        return
        self.TempMsgCount = (Duration // 48) + 1
        self.TempMsgT[1] = Msg
        self.TempMsgDirty = True


    def OnUpdateBeatIndicator(self, Value):
        
        # Blink play button
        SyncLEDMsg = [
            midi.MIDI_NOTEON + (0x5E << 8),
            midi.MIDI_NOTEON + (0x5E << 8) + (0x7F << 16),
            midi.MIDI_NOTEON + (0x5E << 8) + (0x7F << 16),
        ]

        if device.isAssigned():
            device.midiOutNewMsg(SyncLEDMsg[Value], 128)


    def InitMeter(self):

        if device.isAssigned():
            # clear peak indicators
            for m in range(0, len(self.Tracks) - 1):
                device.midiOutMsg(midi.MIDI_CHANAFTERTOUCH + (0xF << 8) + (m << 12))


    def SetNextPage(self):

        newPage = self.Page + 1

        if newPage >= AsparionD400TotalPages:
            newPage = 0

        self.SetPage(newPage)


    def SetPage(self, Value):

        oldPage = self.Page
        self.Page = Value
        self.FirstTrack = 0
        self.SetFirstTrack(self.FirstTrackT[self.FirstTrack])

        if self.IsMain:
            self.AdditionalReceiverCount = device.dispatchReceiverCount()
            self.PrintText("Receiving additional extenders: " + str(self.AdditionalReceiverCount))

            if self.AdditionalReceiverCount != 0:
                for n in range(0, self.AdditionalReceiverCount): # send set page to others
                    device.dispatch(n, midi.MIDI_PROGRAMCHANGE + (0x44 << 8) + (Value << 16))

        self.UpdateTracks()
        self.UpdateTransportLEDs()
        #self.PrintText("Set page " + str(Value))


    def UpdateSelectedTrack(self):

        self.SelectedTrackNoGlobal = mixer.trackNumber()

        if device.isAssigned():
            for m in range(0, len(self.Tracks) - 1):

                if self.Tracks[m].TrackNum == self.SelectedTrackNoGlobal:
                    SelectedTrackNo = m

                self.UpdateTactLED(0x18 + m, self.Tracks[m].TrackNum == self.SelectedTrackNoGlobal)

            if self.Page in [AsparionD400Page_Sends, AsparionD400Page_FX]:
                self.UpdateTracks()

            if (self.IsMain):

                # Update pan & ss
                currentPanID = mixer.getTrackPluginId(mixer.trackNumber(), 0) + midi.REC_Mixer_Pan
                currentPanValue = mixer.getAutoSmoothEventValue(currentPanID)
                currentPanValueAdjusted = int((currentPanValue + 6400) // 100.4)
                device.midiOutMsg(midi.MIDI_CONTROLCHANGE + ((0x72) << 8) + (currentPanValueAdjusted << 16))

                currentSSID = mixer.getTrackPluginId(mixer.trackNumber(), 0) + midi.REC_Mixer_SS
                currentSSValue = mixer.getAutoSmoothEventValue(currentSSID)
                currentSSValueAdjusted = int((currentSSValue + 64) // 1.004)
                device.midiOutMsg(midi.MIDI_CONTROLCHANGE + ((0x73) << 8) + (currentSSValueAdjusted << 16))

                self.UpdateLinkedControllers()


    def UpdateLinkedControllers(self):

        if (self.IsMain == False):
            return

        # Update linked controllers A1 to A8
        self.UpdateLinkedController(0x50, 0)
        self.UpdateLinkedController(0x51, 1)
        self.UpdateLinkedController(0x52, 2)
        self.UpdateLinkedController(0x53, 3)

        self.UpdateLinkedController(0x22, 4)
        self.UpdateLinkedController(0x23, 5)
        self.UpdateLinkedController(0x24, 6)
        self.UpdateLinkedController(0x25, 7)


    def UpdateLinkedController(self, ID, No):

        eventID = device.findEventID(midi.EncodeRemoteControlID(device.getPortNumber(), 0, 0) + ID, 1)
        newVal = int(device.getLinkedValue(eventID) * 127)

        if self.GlobalRingPositions[No] != newVal:
            self.GlobalRingPositions[No] = newVal
            device.midiOutMsg(midi.MIDI_CONTROLCHANGE + ((ID) << 8) + (newVal << 16))
            

    def UpdateFader(self, Num):

        data1 = 0
        data2 = 0
        baseID = 0
        center = 0
        b = False

        if device.isAssigned():
            
            sv = mixer.getEventValue(self.Tracks[Num].SliderEventID)

            if Num < 8:

                # V-Pot
                center = self.Tracks[Num].KnobCenter
                if self.Tracks[Num].KnobEventID >= 0:
                    m = mixer.getEventValue(
                        self.Tracks[Num].KnobEventID, midi.MaxInt, False
                    )
                    if center < 0:
                        if (
                            self.Tracks[Num].KnobResetEventID
                            == self.Tracks[Num].KnobEventID
                        ):
                            center = int(m != self.Tracks[Num].KnobResetValue)
                        else:
                            center = int(sv != self.Tracks[Num].KnobResetValue)

                    if self.Tracks[Num].KnobMode < 2:
                        data1 = 1 + round(m * (10 / midi.FromMIDI_Max))
                    else:
                        data1 = round(m * (11 / midi.FromMIDI_Max))
                    if self.Tracks[Num].KnobMode > 3:
                        data1 = center << 6
                    else:
                        data1 = (
                            data1 + (self.Tracks[Num].KnobMode << 4) + (center << 6)
                        )
                else:
                    Data1 = 0

                # arm, solo, mute

                self.UpdateTactLED(0x00 + Num, int(mixer.isTrackArmed(self.Tracks[Num].TrackNum)) * (1 + int(transport.isRecording())))
                self.UpdateTactLED(0x08 + Num, mixer.isTrackSolo(self.Tracks[Num].TrackNum))
                self.UpdateTactLED(0x10 + Num, not mixer.isTrackEnabled(self.Tracks[Num].TrackNum))


            # fader
            newFaderPos = self.AlphaTrack_LevelToSlider(sv)

            if newFaderPos != self.Tracks[Num].LastSentFaderPos:
                self.Tracks[Num].LastSentFaderPos = newFaderPos
                data1 = newFaderPos
                data2 = data1 & 127
                data1 = data1 >> 7
                device.midiOutNewMsg(
                    midi.MIDI_PITCHBEND + Num + (data2 << 8) + (data1 << 16),
                    self.Tracks[Num].LastValueIndex + 5,
                )

            Dirty = False


    def AlphaTrack_LevelToSlider(self, Value, Max=midi.FromMIDI_Max):

        return round(Value / Max * self.AlphaTrack_SliderMax)


    def AlphaTrack_SliderToLevel(self, Value, Max=midi.FromMIDI_Max):

        return min(round(Value / self.AlphaTrack_SliderMax * Max), Max)


    def UpdateTracks(self):

        f = self.FirstTrackT[self.FirstTrack]
        CurID = mixer.getTrackPluginId(mixer.trackNumber(), 0)

        for m in range(0, len(self.Tracks)):

            self.Tracks[m].KnobPressEventID = -1

            # mixer
            if m == 8:
                self.Tracks[m].TrackNum = -2
                self.Tracks[m].BaseEventID = midi.REC_MainVol
                self.Tracks[m].SliderEventID = self.Tracks[m].BaseEventID
                self.Tracks[m].SliderName = "Master Vol"
            else:
                self.Tracks[m].TrackNum = midi.TrackNum_Master + (
                    (f + m) % mixer.trackCount()
                )
                self.Tracks[m].BaseEventID = mixer.getTrackPluginId(
                    self.Tracks[m].TrackNum, 0
                )
                self.Tracks[m].SliderEventID = (
                    self.Tracks[m].BaseEventID + midi.REC_Mixer_Vol
                )
                s = "" # mixer.getTrackName(self.Tracks[m].TrackNum)
                self.Tracks[m].SliderName = s + ""

                self.Name = mixer.getTrackName(self.Tracks[m].TrackNum) 
                

                self.Tracks[m].KnobEventID = -1
                self.Tracks[m].KnobResetEventID = -1
                self.Tracks[m].KnobResetValue = midi.FromMIDI_Max >> 1
                self.Tracks[m].KnobName = ""
                self.Tracks[m].TextValuePre = ""
                self.Tracks[m].KnobMode = 1  # parameter, pan, volume, off
                self.Tracks[m].KnobCenter = -1
                if self.Page == AsparionD400Page_Volume:
                    self.Tracks[m].KnobName = "Volume"
                    self.Tracks[m].TextValuePre = ""

                    val = mixer.getAutoSmoothEventValue(self.Tracks[m].SliderEventID)
                    s = mixer.getEventIDValueString(self.Tracks[m].SliderEventID, val)
                    self.WriteDisplayPerTrack(s.rjust(7, " "), 1, m)
                elif self.Page == AsparionD400Page_Channels:
                    self.Tracks[m].KnobName = "Channel"
                    self.Name = channels.getChannelName(self.Tracks[m].TrackNum) 
                    self.Tracks[m].Value = int(channels.getChannelVolume(self.Tracks[m].TrackNum) * 100)

                    self.Tracks[m].TextValue = str(self.Tracks[m].Value) + "%"
                    self.Tracks[m].TextValuePre = "Chn"
                    self.WriteDisplayPerTrack(self.Tracks[m].TextValuePre + self.Tracks[m].TextValue.rjust(5, " "), 1, m)
                elif self.Page == AsparionD400Page_Pan:
                    self.Tracks[m].KnobEventID = (self.Tracks[m].BaseEventID + midi.REC_Mixer_Pan)
                    self.Tracks[m].KnobResetEventID = self.Tracks[m].KnobEventID
                    self.Tracks[m].KnobName = "Pan"

                    self.Tracks[m].Value = mixer.getAutoSmoothEventValue(self.Tracks[m].KnobEventID) // 64 # -6400 to 6400 -> -100 to 100
                    self.Tracks[m].TextValue = str(self.Tracks[m].Value) + "%"
                    self.Tracks[m].TextValuePre = "Pan"
                    self.WriteDisplayPerTrack(self.Tracks[m].TextValuePre + self.Tracks[m].TextValue.rjust(5, " "), 1, m)
                elif self.Page == AsparionD400Page_Stereo:
                    self.Tracks[m].KnobEventID = (self.Tracks[m].BaseEventID + midi.REC_Mixer_SS)
                    self.Tracks[m].KnobResetEventID = self.Tracks[m].KnobEventID
                    self.Tracks[m].KnobName = "Stereo"
                    
                    self.Tracks[m].Value = mixer.getAutoSmoothEventValue(self.Tracks[m].KnobEventID) * 100 // 64 # -64 to 64 -> -100 to 100
                    self.Tracks[m].TextValue = str(self.Tracks[m].Value) + "%"
                    self.Tracks[m].TextValuePre = "Ste"
                    self.WriteDisplayPerTrack(self.Tracks[m].TextValuePre + self.Tracks[m].TextValue.rjust(5, " "), 1, m)
                elif self.Page == AsparionD400Page_Sends:
                    self.Tracks[m].KnobEventID = (CurID + midi.REC_Mixer_Send_First + self.Tracks[m].TrackNum)
                    s = mixer.getEventIDName(self.Tracks[m].KnobEventID)
                    self.Tracks[m].KnobName = "Send"
                    self.Tracks[m].KnobResetValue = round(12800 * midi.FromMIDI_Max / 16000)
                    self.Tracks[m].KnobCenter = mixer.getRouteSendActive(mixer.trackNumber(), self.Tracks[m].TrackNum)
                    if self.Tracks[m].KnobCenter == 0:
                        self.Tracks[m].KnobMode = 4
                    else:
                        self.Tracks[m].KnobMode = 2

                    self.Tracks[m].Value = mixer.getAutoSmoothEventValue(self.Tracks[m].KnobEventID) // 160 # 0 to 16000 -> 0 to 100
                    self.Tracks[m].TextValue = str(self.Tracks[m].Value) + "%"
                    self.Tracks[m].TextValuePre = "Sen"
                    self.WriteDisplayPerTrack(self.Tracks[m].TextValuePre + self.Tracks[m].TextValue.rjust(5, " "), 1, m)
                elif self.Page == AsparionD400Page_FX:
                    CurID = mixer.getTrackPluginId(mixer.trackNumber(), m)
                    self.Tracks[m].KnobEventID = CurID + midi.REC_Plug_MixLevel
                    s = mixer.getEventIDName(self.Tracks[m].KnobEventID)
                    self.Tracks[m].KnobName = "FX"
                    self.Tracks[m].KnobResetValue = midi.FromMIDI_Max

                    IsValid = mixer.isTrackPluginValid(mixer.trackNumber(), m)
                    IsEnabledAuto = mixer.isTrackAutomationEnabled(mixer.trackNumber(), m)
                    if IsValid:
                        self.Tracks[m].KnobMode = 2
                        self.Tracks[m].KnobPressEventID = CurID + midi.REC_Plug_Mute
                    else:
                        self.Tracks[m].KnobMode = 4
                    self.Tracks[m].KnobCenter = int(IsValid & IsEnabledAuto)

                    self.Tracks[m].Value = mixer.getAutoSmoothEventValue(self.Tracks[m].KnobEventID) // 128 # 0 to 16000 -> 0 to 100
                    self.Tracks[m].TextValue = str(self.Tracks[m].Value) + "%"
                    self.Tracks[m].TextValuePre = "FX"
                    self.WriteDisplayPerTrack(self.Tracks[m].TextValuePre + self.Tracks[m].TextValue.rjust(5, " "), 1, m)
                elif self.Page == AsparionD400Page_EQ:
                    self.Tracks[m].TextValuePre = ""
                    if m < 3:
                        # gain & freq
                        self.Tracks[m].SliderEventID = (
                            CurID + midi.REC_Mixer_EQ_Gain + m
                        )
                        self.Tracks[m].KnobResetEventID = self.Tracks[m].SliderEventID
                        s = mixer.getEventIDName(self.Tracks[m].SliderEventID)
                        self.Tracks[m].SliderName = s
                        self.Tracks[m].KnobEventID = (
                            CurID + midi.REC_Mixer_EQ_Freq + m
                        )
                        s = mixer.getEventIDName(self.Tracks[m].KnobEventID)
                        self.Tracks[m].KnobName = s
                        self.Tracks[m].KnobResetValue = midi.FromMIDI_Max >> 1
                        self.Tracks[m].KnobCenter = -2
                        self.Tracks[m].KnobMode = 0
                    else:
                        if m < 6:
                            # Q
                            self.Tracks[m].SliderEventID = (
                                CurID + midi.REC_Mixer_EQ_Q + m - 3
                            )
                            self.Tracks[m].KnobResetEventID = self.Tracks[
                                m
                            ].SliderEventID
                            s = mixer.getEventIDName(self.Tracks[m].SliderEventID)
                            self.Tracks[m].SliderName = s
                            self.Tracks[m].KnobEventID = self.Tracks[m].SliderEventID
                            self.Tracks[m].KnobName = self.Tracks[m].SliderName
                            self.Tracks[m].KnobResetValue = 17500
                            self.Tracks[m].KnobCenter = -1
                            self.Tracks[m].KnobMode = 2
                        else:
                            self.Tracks[m].SliderEventID = -1
                            self.Tracks[m].KnobEventID = -1
                            self.Tracks[m].KnobMode = 4

                    val = mixer.getAutoSmoothEventValue(self.Tracks[m].KnobEventID)
                    s = mixer.getEventIDValueString(self.Tracks[m].KnobEventID, val)
                    self.WriteDisplayPerTrack(self.Tracks[m].TextValuePre + s, 1, m)

                if self.Page != AsparionD400Page_Volume: # self.Flip:
                    (
                        self.Tracks[m].KnobEventID,
                        self.Tracks[m].SliderEventID,
                    ) = utils.SwapInt(
                        self.Tracks[m].KnobEventID, self.Tracks[m].SliderEventID
                    )
                    s = self.Tracks[m].SliderName
                    self.Tracks[m].SliderName = self.Tracks[m].KnobName
                    self.Tracks[m].KnobName = s
                    self.Tracks[m].KnobMode = 2
                    if not (
                        self.Page 
                        in [
                            AsparionD400Page_Sends, 
                            AsparionD400Page_FX, 
                            AsparionD400Page_EQ
                        ]
                    ):
                        self.Tracks[m].KnobCenter = -1
                        self.Tracks[m].KnobResetValue = round(12800 * midi.FromMIDI_Max / 16000)
                        self.Tracks[m].KnobResetEventID = self.Tracks[m].KnobEventID

                self.WriteDisplayPerTrack(self.Name, 0, m) # write track names

            self.Tracks[m].LastValueIndex = 48 + m * 6
            self.Tracks[m].Peak = 0
            self.Tracks[m].ZPeak = False
            self.UpdateFader(m)

        self.UpdateSelectedTrack()

    def SetKnobValue(self, Num, Value, Res=midi.EKRes):

        if (self.Tracks[Num].KnobEventID >= 0) & (self.Tracks[Num].KnobMode < 4):
            if Value == midi.MaxInt:
                if self.Page == AsparionD400Page_FX:
                    if self.Tracks[Num].KnobPressEventID >= 0:

                        Value = channels.incEventValue(
                            self.Tracks[Num].KnobPressEventID, 0, midi.EKRes
                        )
                        channels.processRECEvent(
                            self.Tracks[Num].KnobPressEventID, Value, midi.REC_Controller
                        )
                        s = mixer.getEventIDName(self.Tracks[Num].KnobPressEventID)
                        self.OnSendTempMsg(s)
                    return
                else:
                    mixer.automateEvent(
                        self.Tracks[Num].KnobResetEventID,
                        self.Tracks[Num].KnobResetValue,
                        midi.REC_MIDIController,
                        self.SmoothSpeed,
                    )
            else:
                mixer.automateEvent(
                    self.Tracks[Num].KnobEventID,
                    Value,
                    midi.REC_Controller,
                    self.SmoothSpeed,
                    1,
                    Res,
                )

            # hint
            n = mixer.getAutoSmoothEventValue(self.Tracks[Num].KnobEventID)
            s = mixer.getEventIDValueString(self.Tracks[Num].KnobEventID, n)
     
            self.OnSendTempMsg(self.Tracks[Num].TextValuePre + s)


    def SetFirstTrack(self, Value):

        self.FirstTrackT[self.FirstTrack] = (Value + mixer.trackCount()) % mixer.trackCount()
        s = utils.Zeros(self.FirstTrackT[self.FirstTrack], 2, " ")

        device.hardwareRefreshMixerTrack(-1)
                                            
        if (self.IsMain and SelectFirstTrackAtBankChange):
            mixer.setTrackNumber(self.FirstTrackT[self.FirstTrack], midi.curfxMinimalLatencyUpdate | (midi.curfxScrollToMakeVisible if ScrollToFirstTrackAtBankChange else 0))


    def OnUpdateMeters(self):

        for m in range(0, len(self.Tracks) - 1):
            self.Tracks[m].Peak = max(
                self.Tracks[m].Peak,
                round( mixer.getTrackPeaks(self.Tracks[m].TrackNum, midi.PEAK_LR_INV) * self.MeterMax)
            )

    def OnIdle(self):

        # refresh meters
        if device.isAssigned():
            for m in range(0, len(self.Tracks) - 1):
                self.Tracks[m].Tag = utils.Limited(self.Tracks[m].Peak, 0, self.MeterMax)
                self.Tracks[m].Peak = 0
                if self.Tracks[m].Tag == 0:
                    if self.Tracks[m].ZPeak:
                        continue
                    else:
                        self.Tracks[m].ZPeak = True
                else:
                    self.Tracks[m].ZPeak = 0
                device.midiOutMsg(midi.MIDI_CHANAFTERTOUCH + (self.Tracks[m].Tag << 8) + (m << 12))

        # temp message
        if self.TempMsgDirty:
            self.UpdateTempMsg()
            self.TempMsgDirty = False

        if (
            (self.TempMsgCount > 0)
            & (not ui.isInPopupMenu())
        ):
            self.TempMsgCount -= 1
            if self.TempMsgCount == 0:
                self.UpdateTempMsg()


    def UpdateTactLED(self, ID, isOn):

        device.midiOutNewMsg((ID << 8) | (MidiMsgNoteOnLightOn if isOn else MidiMsgNoteOnLightOff), ID + 0xFFF + self.ExNo * 10) # 0xFFF is arbitrary


    def UpdateTransportLEDs(self):

        if device.isAssigned() and self.IsMain:
            # stop
            self.UpdateTactLED(0x5D, transport.isPlaying() == midi.PM_Stopped)
            # loop
            self.UpdateTactLED(0x56, transport.getLoopMode() == midi.SM_Pat)
            # record
            self.UpdateTactLED(0x5F, transport.isRecording())
            # changed flag
            self.UpdateTactLED(0x50, general.getChangedFlag() > 0)
            # metronome
            self.UpdateTactLED(0x59, general.getUseMetronome())
            # rec precount
            self.UpdateTactLED(0x58, general.getPrecount())
            # smoothing
            self.UpdateTactLED(0x33, self.SmoothSpeed > 0)
            # Magic
            self.UpdateTactLED(0x32, self.Page != AsparionD400Page_Volume)
            # snap
            #self.UpdateTactLED(0x56, ui.getSnapMode() != 3)
            # visible windows
            # self.UpdateTactLED(0x78, ui.getVisible(midi.widPlaylist))
            # self.UpdateTactLED(0x7B, ui.getVisible(midi.widMixer))
            # self.UpdateTactLED(0x7A, ui.getVisible(midi.widChannelRack))
            # self.UpdateTactLED(0x79, ui.getVisible(midi.widPianoRoll))
            # self.UpdateTactLED(0x7C, ui.getVisible(midi.widBrowser))
            # self.UpdateTactLED(0x7D, ui.getVisible(5)) # midi.widPlugin, AttributeError: module 'midi' has no attribute 'widPlugin'


    def SetJogSource(self, Value):
        self.JogSource = Value


    def OnWaitingForInput(self):
        pass


AsparionD400 = TAsparionD400()


def OnInit():
    AsparionD400.OnInit()


def OnDeInit():
    AsparionD400.OnDeInit()


def OnDirtyMixerTrack(SetTrackNum):
    AsparionD400.OnDirtyMixerTrack(SetTrackNum)


def OnRefresh(Flags):
    AsparionD400.OnRefresh(Flags)


def OnMidiMsg(event):
    AsparionD400.OnMidiMsg(event)


def OnSendTempMsg(Msg, Duration=1000):
    AsparionD400.OnSendTempMsg(Msg, Duration)


def OnUpdateBeatIndicator(Value):
	AsparionD400.OnUpdateBeatIndicator(Value)


def OnUpdateMeters():
    AsparionD400.OnUpdateMeters()


def OnIdle():
    AsparionD400.OnIdle()


def OnWaitingForInput():
    AsparionD400.OnWaitingForInput()
